package com.ibm.wtp.logger.proxyrender;
/*
 * Licensed Material - Property of IBM (C) Copyright IBM Corp. 2001, 2002 - All Rights Reserved. US
 * Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP
 * Schedule Contract with IBM Corp.
 */

import java.util.logging.Level;

import org.eclipse.core.runtime.*;
import org.eclipse.core.runtime.adaptor.EclipseStarter;

import com.ibm.wtp.common.logger.proxy.*;

/**
 * Insert the type's description here. Creation date: (8/24/2001 9:40:14 AM) @author: Administrator
 */
public abstract class AbstractWorkBenchRenderer implements ILogRenderer2 {
	private boolean fTraceMode = false; // will we actually punch trace messaged or not
	private boolean fSettingTrace = false;
	protected Plugin fMyPlugin = null;
	protected Logger fMyLogger = null;
	protected ILog fWorkBenchLogger = null;

	/**
	 * AbstractWorkBenchRenderer constructor comment.
	 */
	public AbstractWorkBenchRenderer(Logger logger) {
		super();
		fMyLogger = logger;
		fTraceMode = fMyLogger.getTraceMode();

		String pluginID = fMyLogger.getPluginID();
		fMyPlugin = org.eclipse.core.runtime.Platform.getPlugin(pluginID);
		if (fMyPlugin == null)
			throw new RuntimeException("Invalid Plugin ID"); //$NON-NLS-1$

		fWorkBenchLogger = fMyPlugin.getLog();
		setTraceMode(fMyLogger.getTraceMode() || fMyPlugin.isDebugging());
		fMyLogger.setRenderer(this);
	}

	/**
	 * Is the console log for eclipse turned on to sysout. If true, then we shouldn't log
	 * to console anything already logged because Eclipse would of logged it for us.
	 * This comes from the -Declipse.consoleLog="true" which is the default when starting
	 * eclipse from PDE.
	 */
	protected static final boolean consoleLogOn = "true".equals(System.getProperty(EclipseStarter.PROP_CONSOLE_LOG));

	/**
	 * Start/Stop Tracing. Creation date: (8/24/2001 9:40:14 AM)
	 */
	public void setTraceMode(boolean flag) {
		if (fSettingTrace)
			return; // Do not allow cycles

		fSettingTrace = true;
		fTraceMode = flag;
		fMyLogger.setTraceMode(flag);
		fSettingTrace = false;
	}

	// The following methods are for historical renderers in case this has been subclassed outside
	// of wtp.common.
	
	/**
	 * Log a string to the trace. 
	 * @param param
	 * @return
	 * 
	 * @since 1.0.0
	 */
	public abstract String log(String param);

	
	/**
	 * Default one that log a string to the trace given a level.
	 * Default simply passes it to log(String) so that we don't break
	 * old subclasses.
	 * <p>If loggedToWorkbench is true, then it shouldn't be logged to
	 * console if consoleLogOn is true because workbench already logged to
	 * console.
	 * 
	 * @param msg
	 * @param l
	 * 
	 * @since 1.0.0
	 */
	protected void log(String msg, Level l, boolean loggedToWorkbench) {
		log(msg);
	}
	
	/**
	 * Render the status log. Creation date: (8/24/2001 9:40:14 AM)
	 * 
	 * @return description of the log's destination e.g., CONSOLE_DESCRIPTION
	 */
	public String log(String msg, int type) {

		String target = logWorkBench(msg, type);
		if (fTraceMode || target.equals(NOLOG_DESCRIPTION))
			return log(msg);
		else
			return target;
	}
	/**
	 * Render the status log. Creation date: (8/24/2001 9:40:14 AM)
	 * 
	 * @return description of the log's destination e.g., CONSOLE_DESCRIPTION
	 */
	public String logWorkBench(String msg, int type) {

		try {
			int ErrCode;
			if (fWorkBenchLogger != null) {
				switch (type) {
					case (ILogRenderer.LOG_ERROR) :
						ErrCode = IStatus.ERROR;
						break;
					case (ILogRenderer.LOG_WARNING) :
						ErrCode = IStatus.WARNING;
						break;
					case (ILogRenderer.LOG_INFO) :
						ErrCode = IStatus.INFO;
						break;
					case (ILogRenderer.LOG_TRACE) :
						ErrCode = IStatus.OK;
						break;
					default :
						throw new RuntimeException("Invalid Log Type"); //$NON-NLS-1$
				}
				Status status = new Status(ErrCode, fMyPlugin.getDescriptor().getUniqueIdentifier(), IStatus.OK, msg, null);
				fWorkBenchLogger.log(status);
				return WORKBENCH_DESCRIPTION;
			} else
				return NOLOG_DESCRIPTION;
		} catch (Throwable t) {
			return NOLOG_DESCRIPTION;
		}
	}
	
	// Default implentation of the ILogRenderer2 interface.
	protected boolean isLogging(Level level) {
		return fTraceMode || fMyLogger.isLoggingLevel(level);
	}
	
	public static final int[] STATUS_LEVEL;
	public static final Level[] STATUS_LEVEL_LOOKUP;
	public static final Level[] LEVEL_STATUS;
	
	static {
		// Status levels that correspond to the log levels, from finest to none, same indexes as from STATUS_LEVEL_LOOKUP.
		STATUS_LEVEL_LOOKUP = new Level[] {Level.INFO, Level.WARNING, Level.SEVERE};
		STATUS_LEVEL = new int[] { IStatus.INFO, IStatus.WARNING, IStatus.ERROR};

		// Levels that correspond to the IStatus levels.
		int maxID = Math.max(IStatus.OK, Math.max(IStatus.INFO, Math.max(IStatus.WARNING, IStatus.ERROR)));
		LEVEL_STATUS = new Level[maxID+1];
		LEVEL_STATUS[IStatus.OK] = Level.FINE;
		LEVEL_STATUS[IStatus.INFO] = Level.INFO;
		LEVEL_STATUS[IStatus.WARNING] = Level.WARNING;
		LEVEL_STATUS[IStatus.ERROR] = Level.SEVERE;
	}
	
	protected Level getLogLevel(IStatus status) {
		return LEVEL_STATUS[status.getSeverity()];
	}

	protected int getStatusSeverity(Level logLevel) {
		for (int i = 0; i < STATUS_LEVEL_LOOKUP.length; i++) {
			if (STATUS_LEVEL_LOOKUP[i] == logLevel)
				return STATUS_LEVEL[i];
		}
		return IStatus.OK;	// Default to ok.
	}
	
	protected String logWorkbench(String msg, Level level) {
		String result = NOLOG_DESCRIPTION;
		// Test again because we could be here simply due to trace mode, in which case we
		// don't want to workbench log it.
		if (fMyLogger.isLoggingLevel(level)) {
			fMyPlugin.getLog().log(	new Status(
					getStatusSeverity(level),
					fMyPlugin.getDescriptor().getUniqueIdentifier(),
					0,
					msg,
					null));
			result = WORKBENCH_DESCRIPTION;
			if (fTraceMode)
				log(msg, level, true);
		} else if (fTraceMode)
			log(msg, level, false);
		return result;
	}
	
	private String getStatusMsg(IStatus s, Level l) {
		if (s.getException() != null)
			return fMyLogger.getGenericMsg(s.toString() + fMyLogger.fLineSeperator + fMyLogger.exceptionToString(s.getException()), l);
		else
			return fMyLogger.getGenericMsg(s.toString(), l);			
	}
	
	protected String logWorkbench(IStatus s, Level level) {
		if (level == DEFAULT)
			level = getLogLevel(s);
		String result = NOLOG_DESCRIPTION;
		// Test again because we could be here simply due to trace mode, in which case we
		// don't want to workbench log it.
		if (fMyLogger.isLoggingLevel(level)) {
			fMyPlugin.getLog().log(s);
			result = WORKBENCH_DESCRIPTION;
			if (fTraceMode)
				log(getStatusMsg(s, level), level, true);				
		} else if (fTraceMode)
			log(getStatusMsg(s, level), level, false);
		return result;
	}
	
	protected String logWorkbench(Throwable t, Level level) {
		String result = NOLOG_DESCRIPTION;
		// Test again because we could be here simply due to trace mode, in which case we
		// don't want to workbench log it.
		if (fMyLogger.isLoggingLevel(level)) {
			fMyPlugin.getLog().log(	new Status(
					getStatusSeverity(level),
					fMyPlugin.getDescriptor().getUniqueIdentifier(),
					0,
					"Exception thrown.",
					t));
			result = WORKBENCH_DESCRIPTION;
			if (fTraceMode)
				log(fMyLogger.getGenericMsg(fMyLogger.exceptionToString(t), level), level, true);
		} else if (fTraceMode)
			log(fMyLogger.getGenericMsg(fMyLogger.exceptionToString(t), level), level, false);
		return result;
	}	

	/* (non-Javadoc)
	 * @see com.ibm.wtp.common.logger.proxy.ILogRenderer2#log(byte, java.util.logging.Level)
	 */
	public String log(boolean b, Level level) {
		if (level == DEFAULT)
			level = Level.FINEST;
		if (isLogging(level)) 
			return logWorkbench(fMyLogger.getGenericMsg(String.valueOf(b), level), level);
		else
			return NOLOG_DESCRIPTION;
	}
	
	/* (non-Javadoc)
	 * @see com.ibm.wtp.common.logger.proxy.ILogRenderer2#log(byte, java.util.logging.Level)
	 */
	public String log(byte b, Level level) {
		if (level == DEFAULT)
			level = Level.FINEST;
		if (isLogging(level)) 
			return logWorkbench(fMyLogger.getGenericMsg(String.valueOf(b), level), level);
		else
			return NOLOG_DESCRIPTION;
	}

	/* (non-Javadoc)
	 * @see com.ibm.wtp.common.logger.proxy.ILogRenderer2#log(char, java.util.logging.Level)
	 */
	public String log(char c, Level level) {
		if (level == DEFAULT)
			level = Level.FINEST;
		if (isLogging(level))
			return logWorkbench(fMyLogger.getGenericMsg(String.valueOf(c), level), level);
		else
			return NOLOG_DESCRIPTION;
	}

	/* (non-Javadoc)
	 * @see com.ibm.wtp.common.logger.proxy.ILogRenderer2#log(double, java.util.logging.Level)
	 */
	public String log(double d, Level level) {
		if (level == DEFAULT)
			level = Level.FINEST;
		if (isLogging(level))
			return logWorkbench(fMyLogger.getGenericMsg(String.valueOf(d), level), level);
		else
			return NOLOG_DESCRIPTION;
	}

	/* (non-Javadoc)
	 * @see com.ibm.wtp.common.logger.proxy.ILogRenderer2#log(float, java.util.logging.Level)
	 */
	public String log(float f, Level level) {
		if (level == DEFAULT)
			level = Level.FINEST;
		if (isLogging(level))
			return logWorkbench(fMyLogger.getGenericMsg(String.valueOf(f), level), level);
		else
			return NOLOG_DESCRIPTION;
	}

	/* (non-Javadoc)
	 * @see com.ibm.wtp.common.logger.proxy.ILogRenderer2#log(int, java.util.logging.Level)
	 */
	public String log(int i, Level level) {
		if (level == DEFAULT)
			level = Level.FINEST;
		if (isLogging(level))
			return logWorkbench(fMyLogger.getGenericMsg(String.valueOf(i), level), level);
		else
			return NOLOG_DESCRIPTION;
	}

	/* (non-Javadoc)
	 * @see com.ibm.wtp.common.logger.proxy.ILogRenderer2#log(long, java.util.logging.Level)
	 */
	public String log(long l, Level level) {
		if (level == DEFAULT)
			level = Level.FINEST;
		if (isLogging(level))
			return logWorkbench(fMyLogger.getGenericMsg(String.valueOf(l), level), level);
		else
			return NOLOG_DESCRIPTION;
	}

	/* (non-Javadoc)
	 * @see com.ibm.wtp.common.logger.proxy.ILogRenderer2#log(java.lang.Object, java.util.logging.Level)
	 */
	public String log(Object o, Level level) {
		if (o instanceof IStatus)
			return logWorkbench((IStatus) o, level);
		if (level == DEFAULT)
			level = Level.FINEST;
		if (isLogging(level))
			return logWorkbench(fMyLogger.getGenericMsg(String.valueOf(o), level), level);
		else
			return NOLOG_DESCRIPTION;
	}

	/* (non-Javadoc)
	 * @see com.ibm.wtp.common.logger.proxy.ILogRenderer2#log(short, java.util.logging.Level)
	 */
	public String log(short s, Level level) {
		if (level == DEFAULT)
			level = Level.FINEST;
		if (isLogging(level))
			return logWorkbench(fMyLogger.getGenericMsg(String.valueOf(s), level), level);
		else
			return NOLOG_DESCRIPTION;
	}
	
	/* (non-Javadoc)
	 * @see com.ibm.wtp.common.logger.proxy.ILogRenderer2#log(java.lang.Throwable, java.util.logging.Level)
	 */
	public String log(Throwable t, Level level) {
		if (t instanceof CoreException)
			return logWorkbench(((CoreException) t).getStatus(), level);			
		if (level == DEFAULT)
			level = Level.SEVERE;
		if (isLogging(level)) {
				return logWorkbench(t, level);
		} else
			return NOLOG_DESCRIPTION;
	}
	
}
