package com.ibm.wtp.common;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.*;

import com.ibm.wtp.common.logger.proxy.Logger;
import com.ibm.wtp.common.util.WTPCommonUtilPlugin;

/*
 * Licensed Material - Property of IBM 
 * (C) Copyright IBM Corp. 2001, 2002 - All Rights Reserved. 
 * US Government Users Restricted Rights - Use, duplication or disclosure 
 * restricted by GSA ADP Schedule Contract with IBM Corp. 
 */

/**
 * @author mdelder 
 */
public class UIContextDetermination {

	private static final int UNKNOWN = 100;

	public static final String HEADLESS_CONTEXT_LITERAL = "Headless";
	public static final String UI_CONTEXT_LITERAL = "UI";

	public static final int HEADLESS_CONTEXT = 102;
	public static final int UI_CONTEXT = 100;

	private static Map cachedExtensions = null;
	private static int currentContext = UNKNOWN;

	private UIContextDetermination() {
	}

	/**
	 * Returns an instance of a given class based 
	 * on the UI or Headless context.
	 *
	 * @param key
	 * @return
	 * @throws IllegalArgumentException If the key is invalid (e.g. no extension is found for the key)
	 */
	public static Object createInstance(String key) {
		Object result = null;
		if (cachedExtensions == null)
			initExtensions();
		IConfigurationElement contextSensitiveClass = (IConfigurationElement) cachedExtensions.get(key);
		try {
			if (contextSensitiveClass != null)
				result = contextSensitiveClass.createExecutableExtension(UIContextDeterminationRegistryReader.UI_CONTEXT_SENSTIVE_CLASS_CLASSNAME_ATTR);
		} catch (CoreException e) {
			Logger.getLogger().logError(e);
		}
		if (result == null)
			Logger.getLogger().logError("Extension not found for key \"" + key + "\".");
		return result;
	}

	/**
	 * Returns the current context -- determines the value if necessary.
	 * @return
	 */
	public static int getCurrentContext() {
		if (currentContext == UNKNOWN) {
			currentContext = HEADLESS_CONTEXT;
			new UITesterRegistryReader().readRegistry();
		}
		return currentContext;
	}

	/**
	 * Invokes the UIContextDeterminationRegistryReader to cache 
	 * all of the extensions, if necessary.
	 *
	 */
	private static void initExtensions() {
		if (cachedExtensions == null) {
			cachedExtensions = new HashMap();
			new UIContextDeterminationRegistryReader().readRegistry();
		}
	}

	/**
	 * Converts the input to one of UI_CONTEXT or HEADLESS_CONTEXT. Defaults to HEADLESS on invalid input
	 * @param literal
	 * @return
	 */
	private static int convertLiteral(String literal) {
		return (UI_CONTEXT_LITERAL.equals(literal)) ? UI_CONTEXT : HEADLESS_CONTEXT;
	}

	/**
	 * Reads the registration of UI Context-sensitive class extensions and 
	 * initializes the cache of the UIContextDetermination object.
	 * @author mdelder
	 */
	private static class UIContextDeterminationRegistryReader extends RegistryReader {

		public static final String UI_CONTEXT_SENSTIVE_CLASS_ELEMENT = "uiContextSensitiveClass";
		public static final String UI_CONTEXT_SENSTIVE_CLASS_KEY_ATTR = "key";
		public static final String UI_CONTEXT_SENSTIVE_CLASS_CLASSNAME_ATTR = "className";
		public static final String UI_CONTEXT_SENSTIVE_CLASS_CONTEXT_ATTR = "context";

		public UIContextDeterminationRegistryReader() {
			super(Platform.getPluginRegistry(), WTPCommonUtilPlugin.PLUGIN_ID, WTPCommonUtilPlugin.UI_CONTEXT_EXTENSION_POINT);
		}

		/* (non-Javadoc)
		 * @see com.ibm.wtp.common.RegistryReader#readElement(org.eclipse.core.runtime.IConfigurationElement)
		 */
		public boolean readElement(IConfigurationElement element) {
			boolean result = false;
			if (element.getName().equals(UI_CONTEXT_SENSTIVE_CLASS_ELEMENT)) {

				String key = element.getAttribute(UI_CONTEXT_SENSTIVE_CLASS_KEY_ATTR);
				String context = element.getAttribute(UI_CONTEXT_SENSTIVE_CLASS_CONTEXT_ATTR);

				if (!cachedExtensions.containsKey(key) || getCurrentContext() == convertLiteral(context))
					cachedExtensions.put(key, element);
				result = true;
			}
			return result;
		}
	}

	/**
	 * Reads the uiTester extension and instantiate the any of the UITester classes it finds. 
	 * 
	 * The implementation has the side effect that if multiple UITesters are registered,
	 * any of them can trip the currentContext into the UI_CONTEXT state. 
	 * @author mdelder 
	 */
	private static class UITesterRegistryReader extends RegistryReader {

		public static final String UI_TESTER_ELEMENT = "uiTester";
		public static final String UI_TESTER_CLASSNAME_ATTR = "className";

		public UITesterRegistryReader() {
			super(Platform.getPluginRegistry(), WTPCommonUtilPlugin.PLUGIN_ID, WTPCommonUtilPlugin.UI_TESTER_EXTENSION_POINT);
		}

		/* (non-Javadoc)
		 * @see com.ibm.etools.emf.workbench.RegistryReader#readElement(org.eclipse.core.runtime.IConfigurationElement)
		 */
		public boolean readElement(IConfigurationElement element) {
			boolean result = false;
			if (element.getName().equals(UI_TESTER_ELEMENT)) {
				result = true;
				try {
					UITester tester = (UITester) element.createExecutableExtension(UI_TESTER_CLASSNAME_ATTR);
					if (tester.isCurrentContextUI())
						currentContext = UI_CONTEXT;

				} catch (CoreException e) {
					Logger.getLogger().logError(e);
				}
			}
			return result;
		}
	}
}
