
package org.eclipse.stellation.task.client;

import java.util.Set;

import org.eclipse.stellation.task.TaskServiceException;

/**
 * A basic interface for client representations of task objects.
 */
public interface Task {
	/**
	 * Retrieve the identifier for this task
	 * @return the task id
	 */
	public String getTaskID();
	
	/**
	 * Get the change task type.  Valid change task types are determined by a particular
	 * back-end implementation. The set of valid change types can be retrieved by calling
	 * TaskServiceClient.getSupportedTaskTypes().
	 * @return the type of this task.
	 */
	public String getTaskType();
	
	/**
	 * Retrieve the set of resources reserved by this task.
	 * @return the set of reserved resources
	 * @throws TaskServiceException if there is an error retrieving the resources.
	 */
	public Set getReservedResources() throws TaskServiceException;
	
	/**
	 * Add a reservation for a resource to the task
	 * @param resid the ID of the resource to add
	 * @throws ResourceNotFoundException if the resource is not recognized by the service
	 * @throws TaskServicePrivilegeException if the user does not have privileges to
	 * 		modify the task.
	 * @throws TaskServiceException if there is some other error processing the request.
	 */
	public void addReservationForResource(String resid) throws TaskServiceException;
	
	
	/**
	 * Add reservations for a set of resources to the task
	 * @param resids the set of IDs of the resources to add
	 * @throws ResourceNotFoundException if some of the resources are not recognized by the service
	 * @throws TaskServicePrivilegeException if the user does not have privileges to
	 * 		modify the task.
	 * @throws TaskServiceException if there is some other error processing the request.
	 */	
	public void addReservationForResources(Set resids) throws TaskServiceException;
	
	/**
	 * Remove reservations for a reserved resource.
	 * @param resid the ID of the resource to remove.
	 * @throws ResourceNotIncludedException if the resource isn't reserved.
	 * @throws TaskServicePrivilegeException if the user does not have privileges to
	 * 		modify the task.
	 * @throws TaskServiceException if some other error occurs processing the request.
	 */
	public void removeReservationForResource(String resid) throws TaskServiceException;
	
	/**
	 * Remove reservations for a set of reserved resources.
	 * @param resids the set of IDs of resources to remove.
	 * @param ResourceNotIncludedException if any of the resources are not reserved
	 * @throws TaskServicePrivilegeException if the user does not have privileges to
	 * 		modify the task.
	 * @throws TaskServiceException if some other error occurs processing the request.
	 */
	public void removeReservationForResources(Set resids) throws TaskServiceException;
	
	/**
	 * Retrieve the ID of the owner of this task.
	 * @return the ID of the owner of this task.
	 */
	public String getOwnerID();
	
	/** 
	 * Retrieve the short name for this task.
	 * @return the task name
	 */
	public String getName();
	
	/**
	 * Retrieve the detailed description of this task.
	 * @return the description of the task.
	 */
	public String getDescription();
	
	/**
	 * Synchronize this task record with the server. Since we live in an asynchronous environment,
	 * it is possible (even likely) that changes will occur after we have retrieved information about the
	 * task. Instead of drowning all of the clients in notifications for every change, the clients can
	 * request updates whenever they want. Calling this method will ensure that the local task object
	 * is updated to the state of the task on the server at the time the request was received. 
	 * 
	 * Developers of particular implementations can decide whether or not to automatically invoke
	 * this method as a part of most other operations.
	 * @throws TaskServiceException
	 */
	public void syncWithServer() throws TaskServiceException;
	
	/**
	 * Check if a task has been completed. This method should always call syncWithServer to
	 * ensure that its response is as up-to-date as possible.
	 * @return true if the task has been completed.
	 * @throws TaskServiceException
	 */
	public boolean taskComplete() throws TaskServiceException;
	
	/**
	 * Inform the system that the task has been completed. This is assumed to occur simultaneously with
	 * a checkin or discard of the changes in the task.
	 * @param completionInfo an object in implementation specific format which contains information
	 * 		about the task completion. 
	 * @throws org.eclipse.stellation.task.TaskServicePrivilegeException if the user is not the
	 * 		owner of this task.
	 * @throws TaskServiceException if any errors occur while completing the task.
	 */
	public void completeTask(Object completionInfo) throws TaskServiceException;
	
	/**
	 * If taskComplete() returns true, this method will return information about the completion of
	 * the task. If the task has not been completed, it will return null.
	 * @return
	 * @throws TaskServiceException
	 */
	public Object getCompletionInfo() throws TaskServiceException;
}
