package org.eclipse.stellation.task.basicimpl;

import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.eclipse.stellation.task.ResourceNotFoundException;
import org.eclipse.stellation.task.ResourceNotIncludedException;
import org.eclipse.stellation.task.TaskCompletionException;
import org.eclipse.stellation.task.TaskConflict;
import org.eclipse.stellation.task.TaskConflictListener;
import org.eclipse.stellation.task.TaskCreationException;
import org.eclipse.stellation.task.TaskNotFoundException;
import org.eclipse.stellation.task.TaskService;
import org.eclipse.stellation.task.TaskServicePrivilegeException;
import org.jdom.Element;

/**
 * A generic implementation of the task service. This version manages the task service
 * completely in memory, but includes calls to abstract functions for storing and retrieving
 * task information.
 * 
 * @author mcc
 *
 */
public  abstract class BasicTaskServiceImpl implements TaskService {
	

	/**
	 * Create a task service object.
	 * @param serviceid a unique name used to identify the task service. This name will be used
	 *       by implementations of the storage procedures to determine where to go to locate
	 *      the task service data at startup. A given task service must always have the same
	 * 	name, even in different server invocations.
	 * @param baseName a globally unique identifier prefix that guarantees that all IDs will
	 * 		be unique. This identifier does <i>not</i> need to be constant between server
	 * 	    invocations.
	 * @throws IOException if there is an error retrieving the task service state.
	 */
	public BasicTaskServiceImpl(String serviceid, String baseName) throws IOException {
		_serviceID = serviceid;
		_taskServiceBaseName = baseName;
		retrieveTaskServiceState();
	}

	/**
	 * Create a new task managed by the task service. The parameter is an XML element
	 * containing data describing the task. For this implementation, the element will look
	 * like the following: 
	 * 	    &lt;TaskInfo userid="id of user to own the task" 
	 *                              name="a simple descriptive task name" 
	 *                              kind="a string describing the type of task"
	 *                              refkey="..."&gt;
	 *            &lt;Description&gt; ... text describing task in detail &lt;/Description&gt;
     *         &lt;/TaskInfo&gt;
     * 
     * <ul>
     * <li> The name is a short name for the task; 
     * <li> the kind is a string identifying the kind of change task; at least "feature", "bug", and "refactor"
     *      are supported.
     * <li> the refkey is an identifier used to connect the
     * task record to some other representation of the task (e.g., a reference number for a bugzilla entry);
     * The refkey can be null.
     * <li> the description is a body of text describing the task.
     * </ul>
     *    
     * @param taskInfo an XML element containing information about the task to create.
	 * @see org.eclipse.stellation.task.TaskService#createNewTask(org.jdom.Element)
	 */
	public String createNewTask(String userid, Element taskInfo)
				throws TaskCreationException {
		if (!taskInfo.getName().equals("TaskInfo"))
			throw new TaskCreationException("Invalid parameter: create operation requires a top level TaskInfo element");
		String taskName = taskInfo.getAttributeValue("name");
		if (taskName == null || taskName.equals(""))
			throw new TaskCreationException("Invalid parameter: TaskInfo element must have a name attribute");
		String kind = taskInfo.getAttributeValue("kind");
		if (kind == null || kind.equals(""))
			throw new TaskCreationException("Invalid parameter: TaskInfo element must have a kind attribute");
		String refkey = taskInfo.getAttributeValue("refkey");
		String description = "<no description>";
		Element descriptionEl = taskInfo.getChild("Description");
		if (descriptionEl != null) {
			description = descriptionEl.getTextNormalize();
		}
		return createNewTask(taskName, userid, kind, refkey, description);
	}
	
	/**
	 * Create a task.
	 * @param name a short name for the task
	 * @param ownerid the user who will own the task (can be null for an ownerless task)
	 * @param kind the task kind 
	 * @param refKey a cross-reference key for the task.
	 * @param description a description of the task.
	 * @return
	 * @throws TaskCreationException
	 */
	public String createNewTask(String name, String ownerid, String kind, String refKey, String description) 
				throws TaskCreationException {
		if (!supportedTaskKind(kind))
			throw new TaskCreationException("Unsupported task kind: " + kind);	
		BasicTask task = new BasicTask(generateUniqueID(), name, kind,  description);
		task.setOwner(ownerid);
		task.setCrossRef(refKey);
		_taskMap.put(task.getID(), task);
		return task.getID();
	}
	
	

	/**
	 * @see org.eclipse.stellation.task.TaskService#addResourceToTask(java.lang.String, java.lang.String)
	 */
	public void addResourceToTask(String userid, String taskid, String resourceId)
				throws TaskNotFoundException, ResourceNotFoundException, TaskServicePrivilegeException {
		BasicTask task = (BasicTask)_taskMap.get(taskid);
		if (task == null)
			throw new TaskNotFoundException(taskid);
		if (task.getOwner() != null && !task.getOwner().equals(userid))
			throw new TaskServicePrivilegeException("User " + userid +
							 " does not have privileges to modify task " + task.getName());
		task.addResource(resourceId);
		// add the reservation for the resource.
		Set reservations = (Set)_resourceReservations.get(resourceId);
		if (reservations == null) {
			reservations = new TreeSet();
			_resourceReservations.put(resourceId, reservations);
		}
		// Check for conflicts: if the reservations for the resource are not empty, then
		// there is a conflict.
		if (!reservations.isEmpty())	{
			Set conflictingTasks = new TreeSet();
			conflictingTasks.addAll(reservations);
			Set conflictingResources = new TreeSet();
			conflictingResources.add(resourceId);
			String conflictID = generateUniqueID();
			
			// Generate outgoing conflicts for each task in conflict
			Iterator conflicts = conflictingTasks.iterator();
			while (conflicts.hasNext()) {
				String conflictingTaskID = (String)conflicts.next();
				TaskConflict conflict = new TaskConflict(conflictID, conflictingTaskID, 
																			taskid,  conflictingTasks, conflictingResources);				
				signalIncomingConflict(conflict);
			}
			TaskConflict outgoingConflict = new TaskConflict(conflictID, taskid, taskid, 
																			conflictingTasks, conflictingResources);
			signalOutgoingConflict(outgoingConflict);
		}
		reservations.add(taskid);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stellation.task.TaskService#addResourcesToTask(java.lang.String, java.util.List)
	 */
	public void addResourcesToTask(String userid, String taskid, Set resourceIds)
			throws TaskNotFoundException, ResourceNotFoundException, TaskServicePrivilegeException {
		BasicTask task = (BasicTask)_taskMap.get(taskid);
		if (task == null)
			throw new TaskNotFoundException(taskid);
		if (task.getOwner() != null && !task.getOwner().equals(userid))
			throw new TaskServicePrivilegeException("User " + userid +
								 " does not have privileges to modify task " + task.getName());			
		Set conflictingTasks = new TreeSet();
		Set conflictingResources = new TreeSet();
		
		// Go through the resources to be added. Add them to the task, and to the reservations
		// records. If there are other reservations for any resource, add the tasks with reservations the resource,
		// and the resource to the conflict lists.
		Iterator rit = resourceIds.iterator();
		while (rit.hasNext()) {
			String rid = (String)rit.next();
			Set reservations = (Set)_resourceReservations.get(rid);
			if (reservations == null) {
				reservations = new TreeSet();
				_resourceReservations.put(rid, reservations);
			}
			if (!reservations.isEmpty()) {
				conflictingResources.add(rid);
				conflictingTasks.addAll(reservations);
			}
			reservations.add(taskid);
		}
		
		// Now, the resources have been added to the task; and the task has reservations for all the
		// resources. Now, if there were any conflicts, generate the conflict list.
		if (!conflictingTasks.isEmpty()) {
			String conflictID = generateUniqueID();
			Iterator taskIt = conflictingTasks.iterator();
			while (taskIt.hasNext()) {
				String conflictingTask = (String)taskIt.next();
				TaskConflict conflict = new TaskConflict(conflictID, conflictingTask, taskid, conflictingTasks, conflictingResources);
				signalIncomingConflict(conflict);
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stellation.task.TaskService#removeResourceFromTask(java.lang.String, java.lang.String)
	 */
	public void removeResourceFromTask(String userid, String taskid, String resourceId)
		throws
			TaskNotFoundException,
			ResourceNotFoundException,
			ResourceNotIncludedException, TaskServicePrivilegeException {
		BasicTask task = (BasicTask)_taskMap.get(taskid);
		if (task == null)
			throw new TaskNotFoundException(taskid);
		if (task.getOwner() != null && !task.getOwner().equals(userid))
			throw new TaskServicePrivilegeException("User " + userid +
								 " does not have privileges to modify task " + task.getName());			
		task.removeResource(resourceId);
		Set reservations = (Set)_resourceReservations.get(resourceId);
		if (reservations == null)
			throw new ResourceNotIncludedException(taskid, resourceId);

	}

	/* (non-Javadoc)
	 * @see org.eclipse.stellation.task.TaskService#removeResourcesFromTask(java.lang.String, java.util.List)
	 */
	public void removeResourcesFromTask(String userid, String taskid, List resourceIds)
		throws
			TaskNotFoundException, ResourceNotFoundException, 
			ResourceNotIncludedException, TaskServicePrivilegeException {
		Iterator resources = resourceIds.iterator();
		Set errors = new HashSet();
		while (resources.hasNext()) {
			String resID = (String)resources.next();
			try {
				removeResourceFromTask(userid, taskid, resID);
			}
			catch (ResourceNotIncludedException rne) {
				errors.add(resID);
			}
		}
		if (!errors.isEmpty()) 
			throw new ResourceNotIncludedException(taskid, errors);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stellation.task.TaskService#completeTask(java.lang.String, org.jdom.Element)
	 */
	public void completeTask(String userid, String taskid, Element completionInfo)
		throws TaskNotFoundException, TaskCompletionException, TaskServicePrivilegeException {
			BasicTask task = (BasicTask)_taskMap.get(taskid);
			if (task == null)
				throw new TaskNotFoundException(taskid);
			if (task.getOwner() != null && !task.getOwner().equals(userid))
				throw new TaskServicePrivilegeException("User " + userid +
								 " does not have privileges to modify task " + task.getName());			
			task.completeTask(completionInfo);
			
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stellation.task.TaskService#addTaskConflictListener(org.eclipse.stellation.task.TaskConflictListener)
	 */
	public void addTaskConflictListener(TaskConflictListener listener) {
		_conflictListeners.add(listener);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stellation.task.TaskService#removeTaskConflictListener(org.eclipse.stellation.task.TaskConflictListener)
	 */
	public void removeTaskConflictListener(TaskConflictListener listener) {
		_conflictListeners.remove(listener);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stellation.task.TaskService#signalIncomingConflict(org.eclipse.stellation.task.TaskConflict)
	 */
	public void signalIncomingConflict(TaskConflict conflict) {
		Iterator listeners = _conflictListeners.iterator();
		while (listeners.hasNext()) {
			TaskConflictListener l = (TaskConflictListener)listeners.next();
			l.handleIncomingConflict(conflict);
		}
	}

	/* 
	 * 
	 */
	public void signalOutgoingConflict(TaskConflict conflict) {
		Iterator listeners = _conflictListeners.iterator();
		while (listeners.hasNext()) {
			TaskConflictListener l = (TaskConflictListener)listeners.next();
			l.handleOutgoingConflict(conflict);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.stellation.task.TaskService#getTaskDescription(java.lang.String, java.lang.String)
	 */
	public String getTaskDescription(String userid, String taskID)
		throws TaskNotFoundException {
		BasicTask task = (BasicTask)_taskMap.get(taskID);
		if (task == null)
			throw new TaskNotFoundException(taskID);
		return task.getDescription();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stellation.task.TaskService#getTaskName(java.lang.String, java.lang.String)
	 */
	public String getTaskName(String userid, String taskID)
		throws TaskNotFoundException {
			BasicTask task = (BasicTask)_taskMap.get(taskID);
			if (task == null)
				throw new TaskNotFoundException(taskID);
			return task.getName();			
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stellation.task.TaskService#getTaskReservedResources(java.lang.String, java.lang.String)
	 */
	public Set getTaskReservedResources(String userid, String taskID)
		throws TaskNotFoundException, TaskServicePrivilegeException {
			BasicTask task = (BasicTask)_taskMap.get(taskID);
			if (task == null)
				throw new TaskNotFoundException(taskID);
			return task.getResources();			
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stellation.task.TaskService#getTasksReservingResource(java.lang.String, java.lang.String)
	 */
	public Set getTasksReservingResource(String userid, String resourceid)
		throws ResourceNotFoundException {
		Set tasks = (Set)_resourceReservations.get(resourceid);
		if (tasks == null || tasks.isEmpty())
			return null;
		else
			return tasks;
	}
	
	public abstract void storeTaskServiceState() throws IOException;
	
	public abstract void retrieveTaskServiceState() throws IOException;
	
	public abstract void storeTask(String taskid) throws IOException;
	
	public abstract void retrieveTask(String taskid) throws IOException, TaskNotFoundException;
	
	

	/**
	 * Check if a given kind of task is supported by this task service. Handling it through a method
	 * like this allows subclasses to extend the set of supported task types.
	 * @param kind a string containing the name of a task type.
	 * @return true if the task type is supported.
	 */
	protected boolean supportedTaskKind(String kind) {
		for (int i=0; i < SupportedTaskTypes.length; i++)
			if (SupportedTaskTypes[i].equals(kind))
				return true;
		return false;
	}
	
	
	
	protected String generateUniqueID() {
		_maxAssignedID++;
		return _taskServiceBaseName + "@@" + _maxAssignedID;
	}
	
	static final public String SupportedTaskTypes[] = new String[] { "feature", "bug", "refactor" };
	
	/**
	 * A map from resource identifiers to sets of tasks that reserve them.
	 */
	protected Map _resourceReservations = new HashMap();
	
	protected Map _taskMap = new HashMap();
	protected int _maxAssignedID = 0;
	protected String _serviceID;
	protected String _taskServiceBaseName;
	
	protected Set _conflictListeners = new HashSet();


}
