package org.eclipse.stellation.task.basicimpl;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.stellation.task.ResourceNotIncludedException;
import org.eclipse.stellation.task.TaskCompletionException;
import org.jdom.Element;

/**
 * A very minimal task implementation.
 */
public class BasicTask {
	
	/**
	 * Create a task record
	 * @param id the task id
	 * @param name a short name for the task
	 * @param kind the type of the task
	 * @param description a longer description of the task.
	 */
	public BasicTask(String id, String name, String kind, String description) {
		_id = id;
		_kind = kind;
		_name = name;
		_xref = null;
		_desc = description;		
	}
	
	/**
	 * Set the owner of the task. If the task has no owner, anyone can modify it. If the owner is set,
	 * only the owner can modify it.
	 * @param userid
	 */
	public void setOwner(String userid) {
		_owner= userid;
	}
	
	/**
	 * Set a cross-reference key for the task. When working with another piece of software that
	 * also uses tasks, this allows a key to be associated with this task to allow it to be
	 * correlated with its corresponding data in the other piece of software.
	 * @param ref a cross-reference key.
	 */
	public void setCrossRef(String ref) {
		_xref = ref;
	}
	
	/**
	 * Get the task identifier
	 * @return a globally unique identifier for the task.
	 */
	public String getID() {
		return _id;
	}
	
	/**
	 * Get the owner of the task
	 * @return the userid of the owner, or null if no one owns the task.
	 */
	public String getOwner() {
		return _owner;
	}
	
	/**
	 * Retrieve the cross-reference key for this task.
	 * @return the key
	 */
	public String getCrossRef() {
		return _xref;
	}
	
	/**
	 * Retrieve the short name for this task.
	 * @return the name
	 */
	public String getName() {
		return _name;
	}
	
	/**
	 * Retrieve the type of the task.
	 * @return a string containing the name of the task type.
	 */
	public String getKind() {
		return _kind;
	}
	
	/** 
	 * Retrieve a detailed description of a task.
	 * @return the description.
	 */
	public String getDescription() {
		return _desc;
	}
	
	/**
	 * Add a resource to an active task.
	 * @param resID the ID of the resource to add.
	 */
	public void addResource(String resID) {
		_resources.add(resID);
	}
	
	/**
	 * Remove a resource from an active task.
	 * @param resID the ID of the resource to remove
	 * @param resID
	 * @throws ResourceNotIncludedException
	 */
	public void removeResource(String resID) throws ResourceNotIncludedException {
		if (_resources.contains(resID))
			_resources.remove(resID);
		else
			throw new ResourceNotIncludedException(_id, resID);
	}
	
	/**
	 * Check if the task contains a given resource
	 * @param resID the ID of the resource
	 * @return true if the task contains the resource.
	 */
	public boolean containsResource(String resID) {
		return _resources.contains(resID);
	}
	
	/**
	 * Check if the task has been marked completed.
	 * @return true if the task is completed.
	 */
	public boolean taskCompleted() {
		return _taskCompleted;
	}
	
	public Element getCompletionData() {
		return _completion;
	}
	
	/**
	 * Return the set of resources reserved by the task.
	 * @return a set containing the resource IDs.
	 */
	public Set getResources() {
		return _resources;
	}
	
	/**
	 * Mark a task as complete.
	 * @throws TaskCompletionException
	 */
	public void completeTask(Element completionInfo) throws TaskCompletionException {
		_taskCompleted = true;
		_completion = completionInfo;
	}

	protected boolean _taskCompleted = false;

	/**
	 * A globally unique task identifier.
	 */
	protected String _id;
	
	protected String _kind;
	
	/**
	 * A descriptive name for the task.
	 */
	protected String _name;
	
	/**
	 * A key used to cross-reference the task to another task management tool.
	 */
	protected String _xref;
	
	protected String _owner;
	
	protected String _desc;
	
	protected Set _resources = new HashSet();
	
	/**
	 * A place to store information associated with the completion of the task. 
	 */
	protected Element _completion = null;

}
