
package org.eclipse.stellation.task;

import java.util.List;
import java.util.Set;

import org.jdom.Element;

/**
 * The low-level interface for a task service. Implementors of task service back-ends will work with
 * this interface; typical task service clients will use a more convenient, high-level interface.
 * 
 * The task service performs two jobs: it keeps track of what developers are doing, and it identifies when the activities of two 
 * developers may produce a conflict. 
 * 
 * It is built around the notion of a change task. The task service tries to be as agnostic as possible about what
 * a change task is; it keeps track of task identity, retains opaque metadata about the task, and maintains information about
 * what developer is performing the task, and what resources that developer is changing as a part of the task.
 * 
 * The method for tracking resources that are part of a change task is based on <em>change reservations</em>. Before
 * a developer starts to change any resource, they place a reservation for that resource. The reservation indicates that they
 * intend to eventually check in a modified version of that resource.
 * 
 * A potential conflict occurs when two developers working on different change tasks, possibly working on entirely different
 * branches of the system, place change reservations for the same resource. When this occurs, the task service sends a
 * notification to both developers.
 * 
 */
public interface TaskService {
	
	/**
	 * Create a new task to represent a set of changes to be performed.
	 * @param userid the ID of the user creating the task.
	 * @param taskInfo an XML element describing the task to be created. The format of
	 * 		this element is dependent on the particular task service implementation. It should
	 * 		include at least a human-readable short name, and a task description.
	 * @return a globally unique task identifier for the new task.
	 * @throws TaskCreationException if there is an error creating the task.
	 * @throws TaskServicePrivilegeException if the user does not have the privileges to perform this operation
	 */
	public String createNewTask(String userid, Element taskInfo ) 
			throws TaskCreationException, TaskServicePrivilegeException;
	
	/**
	 * Get the short name of a task.
	 * @param the userid of the user requesting the name of the task.
	 * @param taskID the ID of the task whose short name should be retrieved.
	 * @return the task name
	 * @throws TaskNotFoundException
	 */
	public String getTaskName(String userid, String taskID) 
			throws TaskNotFoundException;
	
	/**
	 * Get a description of a task.
	 * @param userid the userid of the user requesting information about the task.
	 * @param taskID the ID of the task
	 * @return the description
	 * @throws TaskNotFoundException
	 */
	public String getTaskDescription(String userid, String taskID) throws TaskNotFoundException;
	
	/**
	 * Retrieve the list of resources which the task has placed reservations on. 
	 * 
	 * The list is accurate as of the time that the request was received by the server driving
	 * the task service. Note that because of communication time, the set of resources may have
	 * changed between the time this result was generated by the server, and received by the client.
	 * @param userid the ID of the user requesting the set of reserved resources.
	 * @param taskID
	 * @return the set of resources reserved by the task.
	 * @throws TaskNotFoundException
	 * @throws TaskServicePrivilegeException if the user does not have the privileges to perform this operation
	 */
	public Set getTaskReservedResources(String userid, String taskID)
		 throws TaskNotFoundException, TaskServicePrivilegeException;
	
	/**
	 * retrieve a list of the tasks that have reservations for a particular resource. If no reservations
	 * exist for the resource, return null.
	 * @param userid the user requesting the list of reservations
	 * @param resourceid the ID of the resource
	 * @return a list of task identifiers, or null if not task has a reservation for the resource
	 */
	public Set getTasksReservingResource(String userid, String resourceid)
			throws ResourceNotFoundException;
	
	/**
	 * Place a change reservation for a resource to be changed in a particular task.
	 * @param userid the ID of the user adding a resource to the task.
	 * @param taskid the ID of the task to add the resource to.
	 * @param resourceId the ID of the resource to add
	 * @throws TaskNotFoundException
	 * @throws ResourceNotFoundException
	 * @throws TaskServicePrivilegeException if the user does not have the privileges to perform this operation
	 */
	public void addResourceToTask(String userid, String taskid, String resourceId) 
			throws TaskNotFoundException, ResourceNotFoundException, TaskServicePrivilegeException;
	
	/**
	 * Place change reservations for a set of resources to be changed in a particular task.
	 * @param userid the ID of the user adding resources to the task. 
	 * @param taskid the ID of the task
	 * @param resourceIds a set containing the IDs of the resources to add.
	 * @throws TaskNotFoundException
	 * @throws ResourceNotFoundException
	 * @throws TaskServicePrivilegeException if the user does not have the privileges to perform this operation
	 */
	public void addResourcesToTask(String userid, String taskid, Set resourceIds)
			throws TaskNotFoundException, ResourceNotFoundException, TaskServicePrivilegeException;
	
	/**
	 * Remove a reservation for a resource from a task.
	 * @param userid the user requesting that the reservation be removed
	 * @param taskid the ID of the task
	 * @param resourceId the ID of the resource to remove
	 * @throws TaskNotFoundException if the task is not found by the server
	 * @throws ResourceNotFoundException if the resourceID is not recognized as corresponding to
	 * 			an existing resource by the server 
	 * @throws ResourceNotIncludedException if the resourceID is a valid ID for a resource, but that
	 * 		resource is not reserved by the task.
	 * @throws TaskServicePrivilegeException if the user does not have the privileges to perform this operation
	 */
	public void removeResourceFromTask(String userid, String taskid, String resourceId)
			throws TaskNotFoundException, ResourceNotFoundException, 
					   ResourceNotIncludedException, TaskServicePrivilegeException;

	/**
	 * Remove reservations for a set of resources from a task.
	 * @param userid the ID of the user requesting that reservations be removed from hte task.
	 * @param taskid the ID of the task
	 * @param resourceIds a set of the IDs of the resources to remove
	 * @throws TaskNotFoundException if the task is not found by the server
	 * @throws ResourceNotFoundException if the resourceID is not recognized as corresponding to
	 * 			an existing resource by the server 
	 * @throws ResourceNotIncludedException if the resourceID is a valid ID for a resource, but that
	 * 		resource is not reserved by the task.
	 * @throws TaskServicePrivilegeException if the user does not have the privileges to perform this operation
	 */					   
	public void removeResourcesFromTask(String userid, String taskid, List resourceIds)
			throws TaskNotFoundException, ResourceNotFoundException, 
					ResourceNotIncludedException, TaskServicePrivilegeException;

	/**
	 * Inform the system that a change task has been completed. This will release all reservations placed
	 * for the task. It is assumed that this is done atomically with the check-in of changes to the underlying
	 * repository.
	 * @param userid the ID of the user completing the task.
	 * @param taskid the ID of the task
	 * @param completionInfo information describing the completion of the task, containing in an
	 * 		XML document element. The information required is dependent on the task management backend.
	 * @throws TaskNotFoundException if the task cannot be found.
	 * @throws TaskCompletionException if there is an error completing the task. Generally, this will be because
	 * 		you do not have the necessary privileges to complete the task.
	 * @throws TaskServicePrivilegeException if the user does not have the privileges to perform this operation
	 */					
	public void completeTask(String userid, String taskid, Element completionInfo) 
		throws TaskNotFoundException, TaskCompletionException, TaskServicePrivilegeException;
		
	/**
	 * Register a listener that will receive notifications of potential conflicts with
	 * other tasks.
	 * @param listener 
	 */
	public void addTaskConflictListener(TaskConflictListener listener);
	
	/**
	 * Remove a listener that was used for conflicts.
	 * @param listener
	 */
	public void removeTaskConflictListener(TaskConflictListener listener);
	
	/**
	 * Signal a potential conflict with other tasks. (Generally, this method will be called as
	 * a part of a task conflict listeners implementation.) An incoming conflict is one where the
	 * receiving task had a reservation for a resource, and another task just added a new reservation
	 * for the same resource.
	 * @param conflict a record describing the conflict.
	 */
	public void signalIncomingConflict(TaskConflict conflict);
	
	/**
	 * Signal a potential conflict with other tasks. (Generally, this method will be called as
	 * a part of a task conflict listeners implementation.) An outgoing conflict is one where the
	 * receiving task had a reservation for a resource, and another task just added a new reservation
	 * for the same resource.
	 * @param conflict a record describing the conflict.
	 */
	public void signalOutgoingConflict(TaskConflict conflicts);

}
