/*
 * Copyright (c) 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Payneau (Mia-Software) - initial API and implementation
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */
package org.eclipse.debug.ui.workflow.internal.engine;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.workflow.Activator;
import org.eclipse.debug.ui.workflow.Messages;
import org.eclipse.debug.ui.workflow.WorkflowModel;
import org.eclipse.debug.ui.workflow.WorkflowModelEvents;
import org.eclipse.debug.ui.workflow.internal.Constants;
import org.eclipse.debug.ui.workflow.model.workflow.Element;
import org.eclipse.debug.ui.workflow.model.workflow.WorkflowFactory;

public class WorkflowModelImpl implements WorkflowModel {

	private List<Element> elementList = new ArrayList<Element>();
	private String rootConfigName;
	
	private final WorkflowModelNotifier notifier = new WorkflowModelNotifier();

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.workflow.WorkflowModelEventsSubscribe#subscribe(org.eclipse.debug.ui.workflow.WorkflowModelEvents)
	 */
	public void subscribe(WorkflowModelEvents listener) {
		this.notifier.subscribe(listener);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.workflow.WorkflowModelEventsSubscribe#unsubscribe(org.eclipse.debug.ui.workflow.WorkflowModelEvents)
	 */
	public void unsubscribe(WorkflowModelEvents listener) {
		this.notifier.unsubscribe(listener);
	}

	private void setRootConfigName(String rootConfigName) {
		this.rootConfigName = rootConfigName;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.workflow.WorkflowModel#getRootConfigName()
	 */
	public String getRootConfigName() {
		return this.rootConfigName;
	}
	
	protected List<Element> getBasicElementsList() {
		return this.elementList;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.workflow.WorkflowModel#getElement(int)
	 */
	public Element getElement(int index) {
		return this.getBasicElementsList().get(index);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.workflow.WorkflowModel#getAllElements()
	 */
	public List<Element> getAllElements() {
		return this.getBasicElementsList();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.workflow.WorkflowModel#addElement(org.eclipse.debug.ui.workflow.model.workflow.Element)
	 */
	public void addElement(Element element) {
		this.getBasicElementsList().add(element);
		this.notifier.notifyElementAdded(element);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.workflow.WorkflowModel#setElement(int, org.eclipse.debug.ui.workflow.model.workflow.Element)
	 */
	public void setElement(int index, Element element) {
		this.getBasicElementsList().set(index, element);
		this.notifier.notifyElementInserted(index, element);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.workflow.WorkflowModel#upElement(int)
	 */
	public void upElement(int indexSelected) {
		Element element = this.getBasicElementsList().get(indexSelected);
		element.setIndex(indexSelected - 1);
		Element alterElement = this.getBasicElementsList().get(indexSelected - 1);
		alterElement.setIndex(indexSelected);
		this.getBasicElementsList().set(indexSelected - 1, element);
		this.getBasicElementsList().set(indexSelected, alterElement);
		
		this.notifier.notifyElementInserted(indexSelected, alterElement);
		this.notifier.notifyElementInserted(indexSelected - 1, element);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.workflow.WorkflowModel#downElement(int)
	 */
	public void downElement(int indexSelected) {
		Element element = this.getBasicElementsList().get(indexSelected);
		element.setIndex(indexSelected + 1);
		Element alterElement = this.getBasicElementsList().get(indexSelected + 1);
		alterElement.setIndex(indexSelected);
		this.getBasicElementsList().set(indexSelected + 1, element);
		this.getBasicElementsList().set(indexSelected, alterElement);
		
		this.notifier.notifyElementInserted(indexSelected, alterElement);
		this.notifier.notifyElementInserted(indexSelected + 1, element);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.workflow.WorkflowModel#removeElement(int)
	 */
	public void removeElement(int index) {
		this.getBasicElementsList().remove(index);
		this.notifier.notifyElementRemoved(index);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.workflow.WorkflowModel#initialization(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	@SuppressWarnings("unchecked")
	public void initialization(ILaunchConfiguration configuration) {
		try {
			this.setRootConfigName(configuration.getName());
			this.elementList = this.stringListToElementList(configuration.getAttribute(
					Constants.attributeWorkflowListMinimalist, new ArrayList<String>()));
			for (Element element : this.getBasicElementsList()) {
				this.notifier.notifyElementAdded(element);
			}
		} catch (CoreException e) {
			IStatus status = new Status(IStatus.ERROR, Activator.PLUGIN_ID, Messages.WorkflowModelImpl_unknownError, e);
			Activator.getDefault().getLog().log(status);
		}
	}

	/**
	 * Allows to transform a string list to element list
	 * 
	 * @param stringList
	 *            String list source
	 * @return Created element list
	 */
	private final List<Element> stringListToElementList(final List<String> stringList) {
		List<Element> elements = new ArrayList<Element>();
		for (int i = 0; i < stringList.size(); i = i + 3) {
			Element element = null;
			// if it'is a Composite
			if (stringList.get(i + 2).equals(
					Constants.launchTypeOfWorkflow)) {
				element = WorkflowFactory.eINSTANCE.createWorkflow();
			} else {
				element = WorkflowFactory.eINSTANCE.createWork();
			}
			element.setName(stringList.get(i));
			element.setIndex(Integer.parseInt(stringList.get(i + 1)));
			element.setType(stringList.get(i + 2));
			elements.add(element);
		}
		return elements;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.workflow.WorkflowModel#save(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void save(final ILaunchConfigurationWorkingCopy configuration) {
		// adjust the name (in the model)
		this.setRootConfigName(configuration.getName());
		// update the list of works (from the model)
		List<String> listString = new ArrayList<String>();
		for (Element element : this.getBasicElementsList()) {
			listString.add(element.getName());
			listString.add(Integer.toString(element.getIndex()));
			listString.add(element.getType());
		}
		configuration.setAttribute(
				Constants.attributeWorkflowListMinimalist,
				listString);
	}

}
