/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/*
 * Copyright (c) 2002 IBM Corp.  All rights reserved.
 * This file is made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 */

#include "LinuxNatives.h"

#include <fam.h>
#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/select.h>

/* Connection to the FAM daemon */
static FAMConnection *fam_connection = NULL;
/* Mutex to control access to the FAM daemon (since only one
   thread can safely access the connection at once) */
static GStaticMutex fam_connection_mutex = G_STATIC_MUTEX_INIT;

#define DEBUG_NATIVES 0

typedef struct {
	char *base_path;
} CallbackData;

typedef enum {
	EVENT_CHANGED = 0,
	EVENT_CREATED = 1,
	EVENT_DELETED = 2,
	EVENT_SHUTDOWN = 3,
	EVENT_OTHER
} EventTypes;

/**
 * Establish a connection to the FAM daemon we use for monitoring
 * if it hasn't already been established.
 *
 * @return FALSE if a connection can't be established, else TRUE
 */
static gboolean
setup_fam (void)
{
	g_static_mutex_lock (&fam_connection_mutex);

	if (fam_connection == NULL) {
		fam_connection = g_new0 (FAMConnection, 1);
		if (FAMOpen(fam_connection) != 0) {
			/* connection failed */
			g_free (fam_connection);
			fam_connection = NULL;
			return FALSE;
		}
	}

	g_static_mutex_unlock (&fam_connection_mutex);

	return TRUE;
}

#if DEBUG_NATIVES

/* Create a new Java string */
static jstring 
new_String (JNIEnv *env, char *string)
{
	return (*env)->NewStringUTF (env, string);
}

/* Print a string using Java's print methods (to get in-order
 * printing) */
static void
PrintString (JNIEnv *env, jclass class, char *cstring)
{
	jstring string;
	jmethodID mid;

	string = new_String (env, cstring);

	mid = (*env)->GetStaticMethodID (env, class, "PrintString", "(Ljava/lang/String;)V");
	if (mid == NULL) {
		printf ("WARNING: failed to print :-(\n");
		return;
	}
	(*env)->CallStaticVoidMethod (env, class, mid, string);
}

#endif

/* Create a new MonitorEvent object in the JVM */
static jobject 
create_new_MonitorEvent (JNIEnv *env, char *filename, int eventType)
{
	jclass monitorEventClass;
	jstring jFilename;
	jmethodID cid;
	jobject result;

	if (filename != NULL) {
		jFilename = (*env)->NewStringUTF (env, filename);
	} else {
		return NULL;
	}

	monitorEventClass = (*env)->FindClass (env, "com/example/autorefresh/linux/MonitorEvent");
	if (monitorEventClass == NULL) {
		return NULL;
	}

	cid = (*env)->GetMethodID (env, monitorEventClass, "<init>", "(Ljava/lang/String;I)V");

	if (cid == NULL) {
		return NULL;
	}

	result = (*env)->NewObject (env, monitorEventClass, cid, jFilename, eventType);

	(*env)->DeleteLocalRef (env, monitorEventClass);

	return result;
}

/*
 * Class:     com_example_autorefresh_linux_LinuxNatives
 * Method:    GetNextEvent
 * Signature: ()[B
 */
JNIEXPORT jstring JNICALL Java_com_example_autorefresh_linux_LinuxNatives_GetNextEvent
  (JNIEnv *env, jclass this)
{
	fd_set readfds;
	int fd;
	FAMEvent event;
	char *path_changed = NULL;
	EventTypes event_type = EVENT_OTHER;
#if DEBUG_NATIVES
	char *message;
#endif

	if (!setup_fam ()) return (jlong) NULL;

#if DEBUG_NATIVES
	g_static_mutex_lock (&fam_connection_mutex);
	message = g_strdup_printf ("GetNextEvent, connection is %d\n", (int)fam_connection);
	g_static_mutex_unlock (&fam_connection_mutex);
	
	PrintString (env, this, message);
	g_free (message);
#endif

	g_static_mutex_lock (&fam_connection_mutex);

	fd = FAMCONNECTION_GETFD (fam_connection);
	FD_ZERO (&readfds);
	FD_SET (fd, &readfds);

	do {
		CallbackData *data;
		if (!FAMPending (fam_connection)) {
			g_static_mutex_unlock (&fam_connection_mutex);
			if (select (fd + 1, &readfds, NULL, NULL, NULL) < 0) {
				/* Uhoh, something went wrong here */
#if DEBUG_NATIVES
				PrintString (env, this, "Hit a very bad place\n");
#endif
				return NULL;
			}
			if (!FD_ISSET (fd, &readfds)) continue;
			g_static_mutex_lock (&fam_connection_mutex);
		}
		
		if (FAMNextEvent(fam_connection, &event) < 0) {
#if DEBUG_NATIVES
			PrintString (env, this, "Hit a very bad place 2\n");
#endif
			/* Uhoh, something went wrong here */
			g_static_mutex_unlock (&fam_connection_mutex);
			return NULL;
		}
		g_static_mutex_unlock (&fam_connection_mutex);

		data = (CallbackData *) event.userdata;
		
		switch (event.code) {
		case FAMChanged:
			event_type = EVENT_CHANGED;
			break;
		case FAMCreated:
			event_type = EVENT_CREATED;
			break;
		case FAMDeleted:
			event_type = EVENT_DELETED;
			break;
			
			/* We don't really care about the following events */
		case FAMStartExecuting:
		case FAMStopExecuting:
		case FAMAcknowledge:
		case FAMExists:
		case FAMEndExist:
		case FAMMoved:
			event_type = EVENT_OTHER;
			break;
		}

		if (event_type == EVENT_CHANGED || event_type == EVENT_CREATED || event_type == EVENT_DELETED) {
			if (event.filename[0] == '/') {
				path_changed = g_strdup (event.filename);
			} else {
				path_changed = g_build_filename (data->base_path, event.filename, NULL);
			}
		}
		
		if (event_type == EVENT_CHANGED && g_file_test (path_changed, G_FILE_TEST_IS_DIR)) {
			/* we ignore changes on directories */
			/* WARNING!!! WE DO THIS BECAUSE THE LAYER ABOVE IS DOING RECURSIVE MONITORING */
			/* WE PROBABLY DON'T WANT THIS ON NON-RECURSIVE */
			event_type = EVENT_OTHER;
			g_free (path_changed);
		}
		//	if (event_type != EVENT_OTHER)
	//printf ("\n********************\n  Filename: %s\n  Base path: %s\n  Final Path: %s\n***********************\n", event.filename, data->base_path, path_changed);
	} while (event_type == EVENT_OTHER);


#if DEBUG_NATIVES
	PrintString (env, this, "Returning from GetNextEvent\n");
#endif

	return create_new_MonitorEvent (env, path_changed, event_type);
}


/*
 * Class:     LinuxNatives
 * Method:    AddDirectoryMonitor
 * Signature: ([B)Ljava/lang/Long;
 */
JNIEXPORT jint JNICALL Java_com_example_autorefresh_linux_LinuxNatives_AddDirectoryMonitor (JNIEnv *env,  jclass this, jstring jPath)
{
	const char *path;
	int result;
	FAMRequest *fam_request;
	jboolean isCopy;
	CallbackData *data;

#if DEBUG_NATIVES
	char *message;

	message = g_strdup_printf ("AddDirectoryMonitor called, handle is %d\n", (int)fam_connection);
	PrintString (env, this, message);
	g_free (message);
#endif

	if (!setup_fam ()) return (jlong) NULL;

	path = (*env)->GetStringUTFChars (env, jPath, &isCopy);

	data = g_new (CallbackData, 1);
	data->base_path = g_strdup (path);

	fam_request = g_new0 (FAMRequest, 1);

#if DEBUG_NATIVES
	message = g_strdup_printf ("Add directory: %s\n", (int)fam_connection, path);
	PrintString (env, this, message);
	g_free (message);*/
#endif
	
	g_static_mutex_lock (&fam_connection_mutex);
	result = FAMMonitorDirectory (fam_connection, path, fam_request, data);
	g_static_mutex_unlock (&fam_connection_mutex);

	(*env)->ReleaseStringUTFChars (env, jPath, path);
	
	return (jint) fam_request;
}

/*
 * Class:     com_example_autorefresh_linux_LinuxNatives
 * Method:    RemoveMonitor
 * Signature: (J)V
 */
JNIEXPORT jboolean JNICALL Java_com_example_autorefresh_linux_LinuxNatives_RemoveMonitor
  (JNIEnv *env, jclass this, jint monitor_handle)
{
	FAMRequest *fam_request = (FAMRequest *) monitor_handle;
	int result;

#if DEBUG_NATIVES
	PrintString (env, this, "RemoveMonitor called\n");
#endif

	if (!setup_fam ()) return FALSE;

	g_static_mutex_lock (&fam_connection_mutex);
	result = FAMCancelMonitor (fam_connection, fam_request);
	g_static_mutex_unlock (&fam_connection_mutex);

	if (result == 0) {
		return TRUE;
	} else {
		return FALSE;
	}
}
