package com.example.autorefresh.ui;

/**********************************************************************
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 * 
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.IWorkbenchWindowActionDelegate;

import com.example.autorefresh.AutoRefreshPlugin;
import com.example.autorefresh.ISearch;
import com.example.autorefresh.ISearchListener;

/**
 * The <code>RefreshActionDelegate</code> provides the user with a way to 
 * start a background refresh of the resources found during a search.  This
 * is only applicable when auto.refresh is <code>false</code> in the 
 * <code>AutoRefreshPlugin</code>.
 * 
 * @see AutoRefreshPlugin
 */
public class RefreshActionDelegate implements IWorkbenchWindowActionDelegate, ISearchListener {
	/**
	 * The action this is a delegate to.
	 */
	private IAction fAction;	
	/**
	 * The window this delegate was created in.
	 */
	private IWorkbenchWindow fWindow;
	
	/**
	 * Starts the refresh in the background.
	 * 
	 * @see IActionDelegate#run(IAction)
	 */
	public void run(IAction action) {
		setStatus(null, "Refreshing resources from local");
		fAction.setEnabled(false);
		Thread t= new Thread(new Runnable() {
			public void run() {
				refresh();
			}
		}, "Background refresh");
		t.start();
	}
	
	/**
	 * Performs the refresh and then updates the state of the button.
	 */
	private void refresh() {
		AutoRefreshPlugin.getDefault().refresh(null);

		Runnable r= new Runnable() {
			public void run() {
				if (fWindow != null && !fWindow.getShell().isDisposed()) {
					setStatus(null, "Refresh complete");
					checkEnabled();
				}
			}
		};

		if (fWindow != null) {
			Shell shell= fWindow.getShell();
			if (shell != null) {
				Display display= shell.getDisplay();
				if (display != null && !display.isDisposed()) {
					display.asyncExec(r);
				}
			}
		}
	}

	/**
	 * @see IWorkbenchWindowActionDelegate#init(IWorkbenchWindow)
	 */
	public void init(IWorkbenchWindow window) {
		fWindow= window;
		AutoRefreshPlugin.getDefault().addSearchListener(this);
	}

	/**
	 * @see IActionDelegate#selectionChanged(IAction, ISelection)
	 */	
	public void selectionChanged(IAction action, ISelection selection) {
		fAction= action;
		checkEnabled();
	}

	/**
	 * @see IWorkbenchWindowActionDelegate#dispose()
	 */
	public void dispose() {
		AutoRefreshPlugin plugin= AutoRefreshPlugin.getDefault();
		if (plugin != null) {
			plugin.removeSearchListener(this);
		}
		fWindow= null;
		fAction= null;
	}	

	/**
	 * @see ISearchListener#searchComplete(ISearch)
	 */
	public void searchComplete(ISearch search) {
		checkEnabled();
	}

	/**
	 * Checks to see if the button should be enabled.
	 */
	private void checkEnabled() {
		if (fAction != null) {
			AutoRefreshPlugin plugin= AutoRefreshPlugin.getDefault();
			if (!plugin.isAutoRefreshing()) {
				boolean enabled= plugin.hasSearchesToRefresh();
				if (fAction.isEnabled() != enabled)
					fAction.setEnabled(enabled);
			} else {
				if (fAction.isEnabled())
					fAction.setEnabled(false);
			}
		}
	}
	
	/**
	 * Sets the status bar to <code>status</code>.
	 */
	private void setStatus(Image image, String status) {
		IWorkbenchPart part= fWindow.getActivePage().getActivePart();
		if (part instanceof IViewPart) {
			IViewPart view= (IViewPart)part;
			IStatusLineManager manager= view.getViewSite().getActionBars().getStatusLineManager();
			manager.setMessage(image, status);
		} else {
			System.out.println(status);
		}	
	}
}



