package com.example.autorefresh.internal;

/**********************************************************************
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 * 
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

import com.example.autorefresh.AutoRefreshPlugin;

/**
 * The <code>SearchManager</code> is responsible for scheduling 
 * searches for resources that may be out of synch with the filesystem.
 */
public class SearchManager implements Runnable {
	/**
	 * State flag indicating if the execution of the manager has been canceled.
	 */
	private boolean fCanceled;
	/**
	 * The current search being performed.
	 */
	private AutoRefreshSearch fSearch;
	/**
	 * The search engine used to peform searches.
	 */
	private SearchEngine fSearchEngine;

	/**
	 * Creates a new search manager.
	 */		
	public SearchManager() {
		fSearch= null;
		fCanceled= false;
	}
	
	/**
	 * Waits for a search, and then performs the search.  If the wait
	 * times out a default search is returned and is executed.
	 */
	public void run() {
		if (AutoRefreshPlugin.getDefault().isDebugging())
			System.out.println("Starting auto-search");
		while(!fCanceled) {
			AutoRefreshSearch search= waitForSearch(AutoRefreshPlugin.getDefault().getSearchDelay());
			if (!fCanceled) {
				if (AutoRefreshPlugin.getDefault().isDebugging())
					System.out.println("Starting search");
				fSearchEngine= new SearchEngine(search);
				fSearchEngine.run();
				fSearchEngine= null;
				if (AutoRefreshPlugin.getDefault().isDebugging())
					System.out.println("Search finished");
			}
		}
	}
	
	/**
	 * Requests that the search manager stop all searches and stop execution.
	 */
	public synchronized void shutdown() {
		fCanceled= true;
		fSearch= null;
		notifyAll();
		if (fSearchEngine != null) {
			fSearchEngine.shutdown();
		}
		fSearchEngine= null;
	}

	/**
	 * Requests that <code>search</code> be carried out.
	 */	
	public synchronized void search(AutoRefreshSearch search) {
		if (search == null)
			return;
		if (fSearchEngine != null) {
			fSearchEngine.search(search);
		} else {
			fSearch= search;
			notifyAll();			
		}
	}

	/**
	 * Waits for a new search.  If no search is set within the given time 
	 * period, returns a default search.
	 */
	private synchronized AutoRefreshSearch waitForSearch(long wait) {
		if (fSearch == null) {
			try {
				wait(wait);
			} catch (InterruptedException e) {
				shutdown();
			}
		}
		if (fSearch != null) {
			AutoRefreshSearch search= fSearch;
			fSearch= null;
			return search;
		} else {
			return AutoRefreshSearch.createDefaultSearch();
		}
	}		
}
