package com.example.autorefresh.internal;

/**********************************************************************
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 * 
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IResource;

import com.example.autorefresh.AutoRefreshPlugin;
import com.example.autorefresh.IScanner;
import com.example.autorefresh.ISearch;

/**
 * The <code>SearchEngine</code> is responsible for carrying out the filesystem
 * searches.
 */
public class SearchEngine implements Runnable {
	/**
	 * State flag indicating if this engine is running.
	 */
	private boolean fIsRunning;
	/**
	 * List of pending searches, contains <code>AutoRefreshSearch</code> objects.
	 */
	private List fSearches;
	/**
	 * The default search
	 */
	private AutoRefreshSearch fDefaultSearch;
	/**
	 * The search currently being executed.
	 */
	private AutoRefreshSearch fCurrentSearch;

	/**
	 * Creates a new engine, and primes it with <code>search</code>.
	 */
	public SearchEngine(AutoRefreshSearch search) {
		fSearches= new ArrayList(10);
		if (search != null) {
			fSearches.add(search);
		}
	}

	/**
	 * Requests that <code>search</code> be executed.
	 */
	public void search(ISearch search) {
		if (!(search instanceof AutoRefreshSearch))
			return;
		
		AutoRefreshSearch s= (AutoRefreshSearch)search;
		if (s.isDefaultSearch()) {
			if (fDefaultSearch == null) {
				fDefaultSearch= s;
			}
		} else {
			fSearches.add(s);
		}
		if (AutoRefreshPlugin.getDefault().isDebugging()) {
			System.out.println("Added search: " + s);
		}
	}

	/**
	 * Iterates over the searches executing one at a time.  Queries the 
	 * <code>AutoRefreshPlugin</code> for the correct scanner to use for 
	 * each resource.
	 */
	public void run() {
		fIsRunning= true;
		while(isRunning() && hasNext()) {
			fCurrentSearch= nextSearch();
			IResource[] resources= fCurrentSearch.getResources();
			
			long time= System.currentTimeMillis();
			if (AutoRefreshPlugin.getDefault().isDebugging()) {
				System.out.println("Search started: " + fCurrentSearch);
			}
			for(int i= 0, length= resources.length; isRunning() && i < length; i++) {
				IResource resource= resources[i];
				IScanner scanner= AutoRefreshPlugin.getDefault().getScanner(resource);
				scanner.scan(resource, fCurrentSearch, fCurrentSearch);
			}
			time= System.currentTimeMillis() - time;
							
			AutoRefreshPlugin.getDefault().searchComplete(fCurrentSearch);
			if (AutoRefreshPlugin.getDefault().isDebugging()) {
				System.out.println("Search complete: " + time + "ms " + fCurrentSearch);
			}
		}
		fIsRunning= false;
	}
	
	/**
	 * Answers <code>true</code> if this engine is currently searching.
	 */
	public boolean isRunning() {
		return fIsRunning;
	}
	
	/**
	 * Requests that the engine stop all searching and shutdown.
	 */
	public void shutdown() {
		fCurrentSearch.cancel();
		fSearches= new ArrayList(0);
		fIsRunning= false;
	}		

	/**
	 * Answers the next <code>AutoRefreshSearch</code>.  If no more searches
	 * exist in the <code>fSearches</code> list, answers the default search.
	 */
	private AutoRefreshSearch nextSearch() {
		if (!fSearches.isEmpty()) {
			return (AutoRefreshSearch)fSearches.remove(0);
		}
		AutoRefreshSearch defaultSearch= fDefaultSearch;
		fDefaultSearch= null;
		return defaultSearch;
	}

	/**
	 * Answers <code>true</code> if there are more searches that need to be
	 * executed.
	 */
	private boolean hasNext() {
		return !fSearches.isEmpty() || fDefaultSearch != null;
	}
	
}
