package com.example.autorefresh.internal;

/**********************************************************************
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 * 
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceDescription;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;

import com.example.autorefresh.AutoRefreshPlugin;

/**
 * The <code>Refresher</code> refreshes resources that have become out of synch
 * with the filesystem.  
 */
public class Refresher {
	/**
	 * Refreshes the resources found in the <code>AutoRefreshSearch</code>
	 * objects contained in <code>searches</code>.
	 * <p>
	 * Turns autobuild off briefly while it refreshes the resources.
	 */
	public void refresh(IProgressMonitor monitor, final List searches) {
		if (searches == null || searches.isEmpty())
			return;

		IWorkspaceRunnable run= new IWorkspaceRunnable() {
			public void run(IProgressMonitor monitor) throws CoreException {
				if (AutoRefreshPlugin.getDefault().isDebugging()) {
					System.out.println("Refresh starting");
				}
				long time= System.currentTimeMillis();
				refreshAll(monitor, searches);
				time= System.currentTimeMillis() - time;
				if (AutoRefreshPlugin.getDefault().isDebugging()) {
					System.out.println("Refresh: " + time);
					System.out.println("Refresh finished");
				}
					
			}

		};
		
		IWorkspace workspace= ResourcesPlugin.getWorkspace();
		boolean isAutoBuilding= workspace.isAutoBuilding();
		try {
			if (isAutoBuilding) {
				setAutoBuilding(workspace, false);
			}			
			workspace.run(run, monitor);
		} catch(CoreException e) {
			e.printStackTrace();
		} finally {
			if (isAutoBuilding) {
				setAutoBuilding(workspace, true);
			}
		}			
		
	}
	
	/**
	 * Refreshes the resources found in the <code>AutoRefreshSearch</code>
	 * objects contained in <code>searches</code>.  Autobuild should be 
	 * turned off before calling this method because this method blocks
	 * the workspace from performing any runnables.
	 */
	private void refreshAll(IProgressMonitor monitor, List searches) {
		for(Iterator i= searches.iterator(); i.hasNext(); ) {
			AutoRefreshSearch search= (AutoRefreshSearch)i.next();
			if (!search.isEmpty()) {
				refresh(monitor, search);
			}
		}
	}
	
	/**
	 * Refreshes the resources found in <code>search</code>.
	 */
	private void refresh(IProgressMonitor monitor, AutoRefreshSearch search) {
		if (AutoRefreshPlugin.getDefault().isDebugging()) {
			System.out.println("Refreshing: " + search);
		}
		if (hasDepthZeroResources(search)) {
			refreshDepthZeroResources(monitor, search);
		}		
		if (hasDepthOneResources(search)) {
			refreshDepthOneResources(monitor, search);
		}
		if (hasDepthInfiniteResources(search)) {
			refreshDepthInfiniteResources(monitor, search);
		}
	}		

	/**
	 * Refreshes the resources that have only changed themselves.
	 */
	private void refreshDepthZeroResources(IProgressMonitor monitor, AutoRefreshSearch search) {
		Set depthZero= getDepthZeroResources(search);
		if (depthZero != null) {
			for(Iterator i= depthZero.iterator(); i.hasNext(); ) {
				IResource resource= (IResource)i.next();
				try {
					resource.refreshLocal(IResource.DEPTH_ZERO, monitor);
				} catch (CoreException e) {
					e.printStackTrace();
				}
			}
		}
	}


	/**
	 * Refreshes the resources that have been added, so they could potentially
	 * need to be refreshed infinite (e.g. <code>IContainer</code> objects).
	 */
	private void refreshDepthOneResources(IProgressMonitor monitor, AutoRefreshSearch search) {
		Map depthOne= getDepthOneResources(search);
		for(Iterator i= depthOne.keySet().iterator(); i.hasNext(); ) {
			IContainer container= (IContainer)i.next();

			// Optimization: only refresh resources that don't already
			// exist, this improves multiple refreshes on the same resource
			List additions= (List)depthOne.get(container);
			for(Iterator j= additions.iterator(); j.hasNext(); ) {
				String name= (String)j.next();
				if (container.findMember(name) != null)
					j.remove();
			}
			try {
				container.refreshLocal(IResource.DEPTH_ONE, monitor);
			} catch(CoreException e) {
				e.printStackTrace();
				return;
			}
			IResource[] members= null;
			try {
				members= container.members();
			} catch(CoreException e) {
				e.printStackTrace();
				return;
			}
			if (members != null) {
				for(Iterator j= additions.iterator(); j.hasNext(); ) {
					String name= (String)j.next();
					IResource member= findMember(members, name);
					if (member instanceof IContainer) {
						try {
							member.refreshLocal(IResource.DEPTH_INFINITE, monitor);
						} catch(CoreException e) {
							e.printStackTrace();
						}
					}
				}
			}
		}
	}

	/**
	 * Refresh the resources that have changed so dramatically that the best
	 * thing to do is refresh the entire tree starting at the given resources.
	 */
	private void refreshDepthInfiniteResources(IProgressMonitor monitor, AutoRefreshSearch search) {
		Set depthInfinite= getDepthInfiniteResources(search);
		if (depthInfinite != null) {
			for(Iterator i= depthInfinite.iterator(); i.hasNext(); ) {
				IResource resource= (IResource)i.next();
				try {
					resource.refreshLocal(IResource.DEPTH_INFINITE, monitor)	;
				} catch (CoreException e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	/**
	 * Answers <code>true</code> if the <code>search</code> contains
	 * resources that need to be refreshed at depth zero.  These resources have
	 * either changed or have been deleted.
	 */
	private boolean hasDepthZeroResources(AutoRefreshSearch search) {
		return search.hasChanges() || search.hasDeletions();
	}
	
	/**
	 * Answers a <code>Set</code> of <code>IResource</code> objects that
	 * need depth zero refreshing.
	 * 
	 * @see #hasDepthZeroResources(AutoRefreshSearch)
	 */
	private Set getDepthZeroResources(AutoRefreshSearch search) {
		Set depthZero= search.getChanges();
		if (depthZero == null) {
			depthZero= search.getDeletions();
		} else {
			Set deletions= search.getDeletions();
			if (deletions != null) {
				depthZero.addAll(deletions);
			}
		}		
		return depthZero;
	}
	
	/**
	 * Answers <code>true</code> if the <code>search</code> contains
	 * resources taht need to be refreshed at depth one.  These resouces
	 * have been added, and could potentially create large changes under
	 * new <code>IContainer</code> objects.
	 */
	private boolean hasDepthOneResources(AutoRefreshSearch search) {
		return search.hasAdditions();
	}
	
	/**
	 * Answers a <code>Map</code> of <code>IResource</code> to <code>List</code>
	 * objects that need depth one refreshing.
	 * 
	 * @see #hasDepthOneResources(AutoRefreshSearch)
	 */
	private Map getDepthOneResources(AutoRefreshSearch search) {
		return search.getAdditions();
	}

	/**
	 * Answers <code>true</code> if the <code>search</code> contains
	 * resources that need to be refreshed at depth infinite.  These resources 
	 * are <code>IFile</code> or <code>IContainer</code> objects that have
	 * become <code>IContainer</code> or <code>IFile</code> objects respectively.
	 */
	private boolean hasDepthInfiniteResources(AutoRefreshSearch search) {
		return search.hasTransforms();
	}
	
	/**
	 * Answers a <code>Set</code> of <code>IResource</code> objects that need 
	 * depth infinite refreshing.
	 * 
	 * @see #hasDepthInfiniteResources(AutoRefreshSearch)
	 */
	private Set getDepthInfiniteResources(AutoRefreshSearch search) {
		return search.getTransforms();
	}
	
	/**
	 * Searches through the <code>members</code> array for a member
	 * whose simple name is <code>name</code>.
	 */
	private IResource findMember(IResource[] members, String name) {
		for (int i = 0; i < members.length; i++) {
			IResource member= members[i];
			if (name.equals(member.getName()))
				return member;
		}
		return null;
	}
	
	/**
	 * Enables or disables autobuilding.
	 */
	private void setAutoBuilding(IWorkspace workspace, boolean autoBuilding) {
		IWorkspaceDescription description= workspace.getDescription();
		description.setAutoBuilding(autoBuilding);
		try {
			workspace.setDescription(description);
		} catch(CoreException e) {
			e.printStackTrace();
		}
	}	
}
