package com.example.autorefresh.internal;

/**********************************************************************
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 * 
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

import java.io.File;

import org.eclipse.core.internal.resources.ResourceInfo;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;

import com.example.autorefresh.IScanner;
import com.example.autorefresh.ISearch;
import com.example.autorefresh.ISearchCollector;

/**
 * The <code>DefaultScanner</code> is an implementation of the <code>IScanner</code>
 * interface that queries the filesystem using the API on <code>java.io.File</code>
 * and compares the results to the state of the workbench.
 */
public class DefaultScanner implements IScanner {
	/**
	 * @see IScanner#scan(IResource, ISearchCollector, ISearch)
	 */
	public void scan(IResource resource, ISearchCollector collector, ISearch search) {
		IPath path= resource.getLocation();
		if (path != null) {
			File file= path.toFile();
			checkResource(resource, file, collector, search);
		}		
	}

	/**
	 * Checks the <code>resource</code> against the <code>file</code>.
	 */
	private void checkResource(IResource resource, File file, ISearchCollector collector, ISearch search) {
		if (resource instanceof IContainer) {
			checkContainer((IContainer)resource, file, collector, search);
		} else {
			if (resource instanceof IFile) {
				checkFile((IFile)resource, file, collector);
			}
		}
	}

	/**
	 * Checks to see if an members of the container have been added, changed, or
	 * deleted or if the container itself has transformed.
	 * 
	 * @see IScanner.scan(IResource, ISearchCollector, ISearch)
	 */
	private void checkContainer(IContainer container, File file, ISearchCollector collector, ISearch search) {
		if (!file.isDirectory()) {
			collector.recordTransform(container);
			return;
		}

		IResource[] members= null;
		try {
			members= container.members();
		} catch (CoreException e) {
			checkContainerExistence(container, file, collector);
			return;
		}
		
		String[] list= file.list();
		if (list != null) {
			int hits= 0;
			for (int i = 0; !search.isCanceled() && i < members.length; i++) {
				IResource member= members[i];
				if (!searchInResource(member, file, list, collector, search)) {
					collector.recordDeletion(member);
				} else {
					hits++;
				}
			}
		
			if (!search.isCanceled() && hits != list.length) {
				for (int i = 0; i < list.length; i++) {				
					if (!resourceExists(list[i], members)) {
						collector.recordAddition(container, list[i]);
					}
				}
			}
		}
	}

	/**
	 * Recursively checks inside of the <code>resource</code>.  Answers
	 * <code>true</code> if the resource was found in the list of resources.
	 */
	private boolean searchInResource(IResource resource, File parent, String[] list, ISearchCollector collector, ISearch search) {
		String name= resource.getName();
		for (int i = 0; i < list.length; i++) {
			if (name.equals(list[i]) && !search.isCanceled()) {
				checkResource(resource, new File(parent, list[i]), collector, search);
				return true;
			}
		}
		return false;	
	}

	/**
	 * Answers <code>true</code> if there exists a resource named <code>name</code>
	 * in the list of resources.
	 */
	private boolean resourceExists(String name, IResource[] resources) {
		for (int i = 0; i < resources.length; i++) {
			if (name.equals(resources[i].getName())) {
				return true;
			}
		}
		return false;	
	}	

	/**
	 * Checks a file to see if it has changed or transformed.
	 * 
	 * @see IScanner.scan(IResource, ISearchCollector, ISearch)
	 */
	private void checkFile(IFile resource, File file, ISearchCollector collector) {
		if (file.isDirectory()) {
			collector.recordTransform(resource);
		}
		if (resource instanceof org.eclipse.core.internal.resources.File) {
			org.eclipse.core.internal.resources.File fileResource= (org.eclipse.core.internal.resources.File)resource;
			ResourceInfo info= fileResource.getResourceInfo(false, false);
			if (info == null) {
				collector.recordDeletion(resource);
				return;
			}
			if (fileResource.isLocal(info.getFlags(), 0)) {
				if (info != null) {
					long timestamp= info.getLocalSyncInfo();
					if (timestamp != file.lastModified()) {
						collector.recordChange(resource);
					}
				}
			} else {
				if (file.exists()) {
					collector.recordChange(resource);
				}
			}
		}
	}

	/**
	 * Checks to see if a container exists.
	 */
	private void checkContainerExistence(IContainer container, File file, ISearchCollector collector) {
		if (!file.exists()) {
			collector.recordDeletion(container);
		}
	}
}
