package com.example.autorefresh.internal;

/**********************************************************************
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 * 
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;

import com.example.autorefresh.ISearch;
import com.example.autorefresh.ISearchCollector;

/**
 * The <code>AutoRefreshSearch</code> is both an <code>ISearch</code>
 * and an <code>ISearchCollector</code> this enables the internal
 * classes to talk about searches and search results with the same object.
 */
public class AutoRefreshSearch implements ISearch, ISearchCollector {
	/**
	 * The resources that will be searched, never <code>null</code>.
	 */
	private IResource[] fResources;
	/**
	 * State flag indicating if the search has been canceled.
	 */
	private boolean fCanceled;
	/**
	 * The map of resource to list of filenames which represent additions.
	 */
	private Map fAdditions;
	/**
	 * The set of deletions.
	 */
	private Set fDeletions;
	/**
	 * The set of changes.
	 */
	private Set fChanges;
	/**
	 * The set of transforms.
	 */
	private Set fTransforms;
	/**
	 * State flag indicating if this search is a default search.  A default
	 * search is one that the the auto search creates when no other searches
	 * are requested, but it is time to perform another search.
	 */
	private boolean fIsDefaultSearch;
	
	/**
	 * Creates a new <code>AutoRefreshSearch</code> for the <code>resources</code>.
	 */
	public AutoRefreshSearch(IResource[] resources) {
		this(resources, false);
	}
	
	/**
	 * Creates a new <code>AutoRefreshSearch</code>, optionally making it a 
	 * default search.  Default searches should be created using the static
	 * method #createDefaultSearch()
	 * 
	 * @see #createDefaultSearch
	 */
	private AutoRefreshSearch(IResource[] resources, boolean isDefault) {
		if (resources == null || resources.length == 0) {
			fResources= getAllProjects();
		} else {
			fResources= resources;
		}
		fIsDefaultSearch= isDefault;
	}

	/**
	 * Queries the workspace for all of the projects.
	 */
	private IResource[] getAllProjects() {
		IWorkspace workspace= ResourcesPlugin.getWorkspace();
		IWorkspaceRoot root= workspace.getRoot();
		if (root != null) {
			return root.getProjects();
		}
		return null;
	}

	/**
	 * Creates a new default search.
	 */		
	static AutoRefreshSearch createDefaultSearch() {
		return new AutoRefreshSearch(null, true);
	}

	/**
	 * @see ISearch#getResources()
	 */
	public IResource[] getResources() {
		return fResources;
	}

	/**
	 * @see ISearch#cancel()
	 */
	public void cancel() {
		fCanceled= true;
	}

	/**
	 * @see ISearch#isCanceled()
	 */
	public boolean isCanceled() {
		return fCanceled;
	}

	/**
	 * @see ISearchCollector#recordAddition(IResource, String)
	 */	
	public void recordAddition(IResource parent, String addition) {
		if (addition == null || parent == null)
			return;
		if (fAdditions == null) {
			fAdditions= new HashMap(5);
		}
		List additions= (List)fAdditions.get(parent);
		if (additions == null) {
			additions= new ArrayList(5);
			fAdditions.put(parent, additions);
		}
		additions.add(addition);
	}
	
	/**
	 * @see ISearchCollector#recordDeletion(IResource)
	 */	
	public void recordDeletion(IResource deletion) {
		if (deletion == null)
			return;
		if (fDeletions == null) {
			fDeletions= new HashSet(5);	
		}
		fDeletions.add(deletion);
	}
	
	/**
	 * @see ISearchCollector#recordChange(IResource)
	 */	
	public void recordChange(IResource change) {
		if (change == null)
			return;
		if (fChanges == null) {
			fChanges= new HashSet(5);
		}
		fChanges.add(change);
	}

	/**
	 * @see ISearchCollector#recordTransform(IResource)
	 */
	public void recordTransform(IResource transform) {
		if (transform == null)
			return;
		if (fTransforms == null) {
			fTransforms= new HashSet(5);
		}
		fTransforms.add(transform);
	}
	
	/**
	 * Answers <code>true</code> if this is a default search.
	 */
	boolean isDefaultSearch() {
		return fIsDefaultSearch;
	}

	/**
	 * Answers the map of additions, or <code>null</code> if the search was
	 * canceled or none were found.
	 */
	Map getAdditions() {
		if (fCanceled) {
			return null;
		}
		return fAdditions;
	}

	/**
	 * Answers the set of deletions, or <code>null</code> if the search was
	 * canceled or none were found.
	 */
	Set getDeletions() {
		if (fCanceled) {
			return null;
		}
		return fDeletions;
	}

	/**
	 * Answers the set of changes, or <code>null</code> if the search was
	 * canceled or none were found.
	 */
	Set getChanges() {
		if (fCanceled) {
			return null;
		}
		return fChanges;
	}

	/**
	 * Answers the set of transforms, or <code>null</code> if the search was
	 * canceled or none were found.
	 */
	Set getTransforms() {
		if (fCanceled) {
			return null;
		}
		return fTransforms;
	}
	
	/**
	 * Answers <code>true</code> if this search has additions and is not canceled.
	 */
	boolean hasAdditions() {
		if (fCanceled) {
			return false;
		}
		return fAdditions != null;
	}

	/**
	 * Answers <code>true</code> if this search has changes and is not canceled.
	 */
	boolean hasChanges() {
		if (fCanceled) {
			return false;
		}
		return fChanges != null;
	}

	/**
	 * Answers <code>true</code> if this search has deletions and is not canceled.
	 */
	boolean hasDeletions() {
		if (fCanceled) {
			return false;
		}
		return fDeletions != null;
	}

	/**
	 * Answers <code>true</code> if this search has transforms and is not canceled.
	 */
	boolean hasTransforms() {
		if (fCanceled) {
			return false;
		}
		return fTransforms != null;
	}

	/**
	 * Answers <code>true</code> if this search does not have any additions,
	 * deletions, changes or transforms.
	 */
	public boolean isEmpty() {
		return !(hasAdditions() || hasChanges() || hasDeletions() || hasTransforms());
	}
}
