package com.example.autorefresh;

/**********************************************************************
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 * 
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IPluginDescriptor;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Plugin;

import com.example.autorefresh.internal.AutoRefreshSearch;
import com.example.autorefresh.internal.DefaultScanner;
import com.example.autorefresh.internal.PathComparator;
import com.example.autorefresh.internal.Refresher;
import com.example.autorefresh.internal.SearchEngine;
import com.example.autorefresh.internal.SearchManager;

/**
 * The <code>AutoRefreshPlugin</code> is responsible for searching the
 * filesystem and performing refreshes on differences it finds between the 
 * filesystem and workspace.
 * <p>
 * NOTE: This plugin will not start automatically, it must be triggered by
 * user events or with client code.  Simply changing a flag is enough to force
 * the plugin to load.
 * <p>
 * The state for the plugin is stored in the prop.ini file located in the 
 * plugin state location.  This file is not created until this plugin has been
 * referenced and the workbench has been shutdown.  There are three values in this 
 * file, they default to:
 * <code>
 * search.delay=60000
 * auto.search=true
 * auto.refresh=true
 * </code>
 * Changing the auto.search and auto.refresh values may also be done in code.
 * <p>
 * The search.delay property dictates the number of milleseconds to wait between
 * searching the filesystem for changes when in auto.search mode.
 * <p>
 * When in auto.search mode, the <code>AutoRefreshPlugin</code> will search 
 * periodically for changes in the filesystem.  When in auto.refresh mode,
 * the <code>AutoRefreshPlugin</code> will perform background refreshing of
 * the changes found. NOTE:  Background refreshing causes all other workbench
 * operations to block, during this time changes can be made to the contents of 
 * buffers, but saves, builds, etc. will block.
 * <p>
 * Clients who wish to give the <code>AutoRefreshPlugin</code> hints on where
 * to search may post a search.  The following is a code snippet that 
 * illustrates a reasonable way to post such a search.
 * <code>
 * AutoRefreshPlugin.getDefault().addAutoRefreshSearchListener(...);
 * IResource[] resources= ...;
 * IAutoRefreshSearch search= AutoRefreshPlugin.getDefault().createSearch(resources);
 * AutoRefreshPlugin.getDefault().search(search);
 * </code>
 * <p>
 * Clients who wish to override the default implementation of the filesystem
 * scanning should register an <code>IScanner</code>.  The registered scanner 
 * will be requested to scan the filesystem whenever the path of the resource
 * to be scanned starts with the path registered with the scanner.  If two
 * scanners could potentially work on a given resource, the one with the most
 * specific path will be chosen.  For example, if <code>FooScanner</code> was
 * registered with the path "c:\foo" and <code>FooBarScanner</code> was 
 * registered with "c:\foo\bar" and the given resource was "c:\foo\bar\baz",
 * <code>FooBarScanner</code> would be selected to perform the scan.
 * <p>
 * Known limititations:
 * <ul>
 * <li>Potential blocking during background refresh, see above.</li>
 * <li>No warning when a project is deleted in the filesystem.</li>
 * </ul>
 * WARNING:  This is experimental API.  It is likely to change in future releases.
 */
public class AutoRefreshPlugin extends Plugin {
	/**
	 * The singleton instance of the <code>AutoRefreshPlugin</code>.
	 */
	private static AutoRefreshPlugin fDefault;

	/**
	 * The search manager schedules searches.
	 */
	private SearchManager fSearchManager;
	/**
	 * When there is no search manager, the search engine is set so that
	 * shutdown may perform quickly.
	 */
	private SearchEngine fSearchEngine;
	/**
	 * The list of completed searches.
	 */
	private List fSearches;
	/**
	 * The registered scanners.
	 */
	private SortedMap fScanners;
	/**
	 * The scanner used when no scanners are registered for a given path.
	 */
	private IScanner fDefaultScanner;
	/**
	 * The search listeners
	 */
	private Set fSearchListeners;
	/**
	 * State flag dictating if auto searching is enabled.
	 */
	private boolean fIsAutoSearching;
	/**
	 * State flag dictating if auto refresh is enabled.
	 */
	private boolean fIsAutoRefreshing;
	/**
	 * State field indicating the time to wait between automatic searches.
	 */
	private long fSearchDelay;

	/**
	 * Property identifier for the auto searching flag
	 */
	private static final String PROP_AUTO_SEARCH= "auto.search";
	/**
	 * Property identifier for the auto refreshing flag
	 */
	private static final String PROP_AUTO_REFRESH= "auto.refresh";
	/**
	 * Property identifier for the search delay field
	 */
	private static final String PROP_SEARCH_DELAY= "search.delay";
	/**
	 * Hardcoded default for search delay
	 */
	public static final long DEFAULT_SEARCH_DELAY= 60 * 1000; // 1 minute
	/**
	 * The name of the property file.
	 */
	public final String PROPERTY_FILE = "prop.ini";
	/**
	 * Creates a new <code>AutoRefreshPlugin</code>.
	 * 
	 * @see Plugin
	 */
	public AutoRefreshPlugin(IPluginDescriptor descriptor) {
		super(descriptor);
		fDefault= this;

		fSearchListeners= new HashSet(5);
		fSearches= new ArrayList(5);
		fScanners= new TreeMap(new PathComparator());		
		fDefaultScanner= new DefaultScanner();

		//setDebugging(true);
		fIsAutoRefreshing= true;
		fIsAutoSearching= true;
		fSearchDelay= DEFAULT_SEARCH_DELAY;
	}

	/**
	 * Answers the singleton instance of the <code>AutoRefreshPlugin</code>.
	 */
	public static AutoRefreshPlugin getDefault() {
		return fDefault;
	}

	/*
	 * @see Plugin#startup()
	 */
	public void startup() throws CoreException {
		loadProperties();
		changeRunLevel();
	}

	/*
	 * @see Plugin#shutdown()
	 */
	public void shutdown() throws CoreException {
		saveProperties();
		
		stopAutoSearch();
		if (fSearchEngine != null) {
			fSearchEngine.shutdown();
			fSearchEngine= null;
		}
	}
	
	/**
	 * Registers a scanner be used to scan the files located
	 * under <code>path</code>.
	 */
	public void registerScanner(IPath path, IScanner scanner) {
		if (path != null && scanner != null) {
			fScanners.put(path, scanner);
		}
	}
	
	/**
	 * Removes the scanner associated with the given <code>path</code>.
	 * This may not force the default scanner to be used as more general
	 * paths may be registered.
	 */
	public void unregisterScanner(IPath path) {
		if (path != null) {
			fScanners.remove(path);
		}
	}
	
	/**
	 * Answers the scanner that would be used to scan the given resource.
	 */
	public IScanner getScanner(IResource resource) {
		if (resource == null)
			return null;
		IPath path= resource.getLocation();
		for(Iterator i= fScanners.keySet().iterator(); i.hasNext(); ) {
			IPath root= (IPath)i.next();
			if (root.isPrefixOf(path))
				return (IScanner)fScanners.get(root);
		}
		return fDefaultScanner;
	}
	
	/**
	 * Request that the given <code>listener</code> receive notification
	 * about searches.
	 */
	public void addSearchListener(ISearchListener listener) {
		if (listener != null)
			fSearchListeners.add(listener);
	}
	
	/**
	 * Request that the given <code>listener</code> no longer receive 
	 * notification about searches.
	 */
	public void removeSearchListener(ISearchListener listener) {
		if (listener != null)
			fSearchListeners.remove(listener);
	}

	/**
	 * Creates a new search on the given resources.  If <code>resources</code>
	 * is <code>null</code> or empty, all projects in the workspace will
	 * be searched.
	 */
	public ISearch createSearch(IResource[] resources) {
		return new AutoRefreshSearch(resources);
	}
	
	/**
	 * Requests that the given search be carried out.  The search
	 * may be canceled midway by calling <code>cancel()</code> on the search 
	 * itself.
	 * <p>
	 * This method returns promptly.  When the search has completed the
	 * registered <code>ISearchListener</code> objects are notified.
	 */
	public void search(ISearch search) {
		if (search == null || !(search instanceof AutoRefreshSearch))
			return;
		AutoRefreshSearch s= (AutoRefreshSearch)search;
		if (isAutoSearching()) {
			fSearchManager.search(s);
		} else {
			if (fSearchEngine == null || !fSearchEngine.isRunning()) {
				fSearchEngine= new SearchEngine(s);
				Thread t= new Thread(fSearchEngine, "Search engine");
				t.start();
			} else {
				fSearchEngine.search(s);
			}
		}
	}
	
	/**
	 * Requests that the resource modifications found since the 
	 * last refresh be refreshed now.  This method blocks until
	 * the refresh is complete.
	 */
	public void refresh(IProgressMonitor monitor) {
		if (!fSearches.isEmpty()) {
			List searches;
			synchronized (fSearches) {
				searches= fSearches;
				fSearches= new ArrayList(5);
			}
			new Refresher().refresh(monitor, searches);
		}
	}
	
	/**
	 * Answers <code>true</code> if non-empty searches have completed
	 * since the last <code>refresh(IProgressMonitor)</code> call was made.
	 */
	public boolean hasSearchesToRefresh() {
		if (fSearches.isEmpty()) {
			return false;
		}
		return true;
	}

	/**
	 * Answers <code>true</code> if the <code>AutoRefreshPlugin</code>
	 * is perfoming periodic automatic searches.
	 */
	public boolean isAutoSearching() {
		return fIsAutoSearching;
	}
	
	/**
	 * Sets the auto searching flag.  If auto searching is enabled, period
	 * searches of the entire contents of the workspace will be performed.
	 * When auto searching is disabled, clients may request that searches be
	 * performed using the <code>search(ISearch)</code> method.
	 */
	public void setAutoSearching(boolean searching) {
		if (fIsAutoSearching != searching) {
			fIsAutoSearching= searching;
			changeRunLevel();
		}
	}
	
	/**
	 * Answers <code>true</code> if the <code>AutoRefreshPlugin</code>
	 * is refreshing resources that are found to be out of synch automatically.
	 */
	public boolean isAutoRefreshing() {
		return fIsAutoRefreshing;
	}
	
	/**
	 * Sets the auto refreshing flag.  When auto refresh is enabled, all
	 * resources found during the search phase will immeadiately be refreshed.
	 * When it is disabled, it is up to a client to call 
	 * <code>refresh(IProgressMonitor)</code> in order to refresh the resources.
	 */
	public void setAutoRefreshing(boolean refreshing) {
		if (fIsAutoRefreshing != refreshing) {
			fIsAutoRefreshing= refreshing;
			changeRunLevel();
		}
	}

	/**
	 * Answers the number of milliseconds the <code>AutoRefreshPlugin</code>
	 * is waiting between searches.
	 */
	public long getSearchDelay() {
		return fSearchDelay;
	}
	
	/**
	 * Sets the search delay field.  The search delay is the amount of time
	 * the auto search will wait between searches.
	 */	
	public void setSearchDelay(long delay) {
		if (delay <= 0)
			delay= DEFAULT_SEARCH_DELAY;
		if (fSearchDelay != delay) {
			fSearchDelay= delay;
		}
	}
	
	/**
	 * For internal use only.  Callback from the search engine that 
	 * a search has completed.
	 */
	public void searchComplete(AutoRefreshSearch search) {
		if (!search.isEmpty()) {
			fSearches.add(search);
			if (isAutoRefreshing()) {
				refresh(null);
			}
		}
		fireSearchComplete(search);
	}
	
	/**
	 * Notifies the listeners that a search has completed.
	 */
	private void fireSearchComplete(ISearch search) {
		if (fSearchListeners.isEmpty())
			return;
		for (Iterator i= fSearchListeners.iterator(); i.hasNext();) {
			((ISearchListener)i.next()).searchComplete(search);
		}
	}
	
	/**
	 * Changes between the states of auto search and auto refresh
	 * enabled and disabled.
	 */
	private void changeRunLevel() {
		if (isAutoSearching()) {
			startAutoSearch();
		} else {
			stopAutoSearch();
		}
	}
	
	/**
	 * Starts the search manager if one hasn't already been created.
	 */
	private void startAutoSearch() {
		if (isAutoSearching() && fSearchManager == null) {
			fSearchManager= new SearchManager();
			Thread t= new Thread(fSearchManager, "Search Manager");
			t.start();
		}
	}
	
	/**
	 * Stops the search manager.
	 */
	private void stopAutoSearch() {
		if (fSearchManager != null) {
			fSearchManager.shutdown();
			fSearchManager= null;
		}
	}
	
	/**
	 * Loads the properties from the plugin state location.
	 */
	private void loadProperties() {
		File file= getPropertyFile();
		Properties properties= new Properties();
		
		if (file.exists()) {
			InputStream in= null;
			try {
				in= new FileInputStream(file); 
				properties.load(in);
			} catch (IOException e) {
				e.printStackTrace();
			} finally {
				if (in != null) {
					try {
						in.close();
					} catch(IOException e) {
						e.printStackTrace();
					}
				}
			}
		}
		
		setAutoRefreshing(getBoolean(properties, PROP_AUTO_REFRESH, true));
		setAutoSearching(getBoolean(properties, PROP_AUTO_SEARCH, true));
		setSearchDelay(getLong(properties, PROP_SEARCH_DELAY, DEFAULT_SEARCH_DELAY));
	}

	/**
	 * Saves the properties to the plugin state location.
	 */	
	private void saveProperties() {
		Properties properties= new Properties();
		properties.put(PROP_AUTO_REFRESH, Boolean.toString(isAutoRefreshing()));
		properties.put(PROP_AUTO_SEARCH, Boolean.toString(isAutoSearching()));
		properties.put(PROP_SEARCH_DELAY, Long.toString(getSearchDelay()));

		OutputStream out= null;
		try {
			out= new FileOutputStream(getPropertyFile());
			properties.store(out, "Auto Refresh Plugin Properties");
		} catch(FileNotFoundException e) {
			e.printStackTrace();
		} catch(IOException e) {
			e.printStackTrace();
		} finally {
			if (out != null) {
				try {
					out.close();
				} catch(IOException e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	/**
	 * Retrieves the named boolean from the properties.  Returns the defaultValue
	 * if the named property cannot be found.
	 */
	private boolean getBoolean(Properties properties, String key, boolean defaultValue) {
		if (properties != null) {
			String p= properties.getProperty(key);
			if (p != null && p.length() > 0) {
				return new Boolean(p).booleanValue();
			}
		}
		return defaultValue;
	}
	
	/**
	 * Retrieves the named long from the properties.  Returns the defaultValue
	 * if the named property cannot be found.
	 */
	private long getLong(Properties properties, String key, long defaultValue) {
		if (properties != null) {
			String p= properties.getProperty(key);
			if (p != null && p.length() > 0) {
				return Long.parseLong(p);
			}
		}
		return defaultValue;
	}

	/**
	 * Answers the <code>java.io.File</code> that represents the property file.
	 */
	private File getPropertyFile() {
		IPath propertiesPath= AutoRefreshPlugin.getDefault().getStateLocation().append(PROPERTY_FILE);
		return propertiesPath.toFile();
	}			
}
