/*
 * Copyright (c) 2007-2012 Niels Provos and Nick Mathewson
 *
 * Copyright (c) 2006 Maxim Yegorushkin <maxim.yegorushkin@gmail.com>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef _MIN_HEAP_H_
#define _MIN_HEAP_H_

#include "mosquitto_broker_internal.h"
#include "mosquitto_internal.h"
#include "memory_mosq.h"

static inline void	     min_heap_ctor(min_heap_t* s);/*db__openʱ*/
static inline void	     min_heap_dtor(min_heap_t* s);/*db__closeʱ*/
static inline void	     min_heap_elem_init(struct mosquitto* context);/*context__initʱ*/
static inline int	     min_heap_elt_is_top(const struct mosquitto *context);
static inline int	     min_heap_elem_greater(struct mosquitto *a, struct mosquitto *b);
static inline int	     min_heap_empty(min_heap_t* s);
static inline unsigned	     min_heap_size(min_heap_t* s);
static inline struct mosquitto*  min_heap_top(min_heap_t* s);
static inline int	     min_heap_reserve(min_heap_t* s, unsigned n);
static inline int	     min_heap_push(min_heap_t* s, struct mosquitto* context);/*context__init / packet__readʱ*/
static inline struct mosquitto*  min_heap_pop(min_heap_t* s); /*main_loop*/
static inline int	     min_heap_erase(min_heap_t* s, struct mosquitto* context);/*packet__readʱ*/
static inline void	     min_heap_shift_up_(min_heap_t* s, unsigned hole_index, struct mosquitto* context);
static inline void	     min_heap_shift_down_(min_heap_t* s, unsigned hole_index, struct mosquitto* context);

int min_heap_elem_greater(struct mosquitto *a, struct mosquitto *b)
{
	return a->keepalive_deadline > b->keepalive_deadline;
}

void min_heap_ctor(min_heap_t* s) { s->p = mosquitto__malloc(sizeof(struct mosquitto*) * MAX_EVENTS); s->n = 0; s->a = MAX_EVENTS; }
void min_heap_dtor(min_heap_t* s) { if (s->p) mosquitto__free(s->p); }
void min_heap_elem_init(struct mosquitto* context) { context->min_heap_idx = -1; }
int min_heap_empty(min_heap_t* s) { return 0u == s->n; }
unsigned min_heap_size(min_heap_t* s) { return s->n; }
struct mosquitto* min_heap_top(min_heap_t* s) { return s->n ? *s->p : NULL; }

int min_heap_push(min_heap_t* s, struct mosquitto* context)
{
    log__printf(NULL, MOSQ_LOG_ERR, "[MINHEAPPUSH]total:%d nodes, push: %s", s->n, context->id);
	//if (min_heap_reserve(s, s->n + 1))
		//return -1;
	if(s->a < s->n) exit(-1);
	min_heap_shift_up_(s, s->n++, context);
	return 0;
}

struct mosquitto* min_heap_pop(min_heap_t* s)
{
	if (s->n)
	{
		struct mosquitto* context = *s->p;
		min_heap_shift_down_(s, 0u, s->p[--s->n]);
		context->min_heap_idx = -1;
		return context;
	}
	return NULL;
}

int min_heap_elt_is_top(const struct mosquitto *context)
{
	return context->min_heap_idx == 0;
}

int min_heap_erase(min_heap_t* s, struct mosquitto* context)
{
    log__printf(NULL, MOSQ_LOG_ERR, "[MINHEAPERASE]total:%d nodes, erase: %s", s->n, context->id);
	if (-1 != context->min_heap_idx)
	{
		struct mosquitto *last = s->p[--s->n];
		unsigned parent = (context->min_heap_idx - 1) / 2;
		/* we replace e with the last element in the heap.  We might need to
		   shift it upward if it is less than its parent, or downward if it is
		   greater than one or both its children. Since the children are known
		   to be less than the parent, it can't need to shift both up and
		   down. */
		if (context->min_heap_idx > 0 && min_heap_elem_greater(s->p[parent], last))
			min_heap_shift_up_(s, context->min_heap_idx, last);
		else
			min_heap_shift_down_(s, context->min_heap_idx, last);
		context->min_heap_idx = -1;
		return 0;
	}
	return -1;
}

int min_heap_reserve(min_heap_t* s, unsigned n)
{
	if (s->a < n)
	{
		struct mosquitto** p;
		unsigned a = s->a ? s->a * 2 : 8;
		if (a < n)
			a = n;
		if (!(p = (struct mosquitto**)mosquitto__realloc(s->p, a * sizeof(*p))))
			return -1;
		s->p = p;
		s->a = a;
	}
	return 0;
}

void min_heap_shift_up_(min_heap_t* s, unsigned hole_index, struct mosquitto* context)
{
    unsigned parent = (hole_index - 1) / 2;
    while (hole_index && min_heap_elem_greater(s->p[parent], context))
    {
	    (s->p[hole_index] = s->p[parent])->min_heap_idx = hole_index;
	    hole_index = parent;
	    parent = (hole_index - 1) / 2;
    }
    (s->p[hole_index] = context)->min_heap_idx = hole_index;
}

void min_heap_shift_down_(min_heap_t* s, unsigned hole_index, struct mosquitto* context)
{
    unsigned min_child = 2 * (hole_index + 1);
    while (min_child <= s->n)
	{
	    min_child -= min_child == s->n || min_heap_elem_greater(s->p[min_child], s->p[min_child - 1]);
	    if (!(min_heap_elem_greater(context, s->p[min_child])))
	        break;
	    (s->p[hole_index] = s->p[min_child])->min_heap_idx = hole_index;
	    hole_index = min_child;
	    min_child = 2 * (hole_index + 1);
	}
    (s->p[hole_index] = context)->min_heap_idx = hole_index;
}

#endif /* _MIN_HEAP_H_ */
