/**********************************************************************
Copyright (c) 2003 IBM Corporation and others.
All rights reserved.  This program and the accompanying materials
are made available under the terms of the Common Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/cpl-v10.html

Contributors:
 IBM Corporation - initial implementation
 IBM Rational - initial implementation
	SCAPA Technologies - initial implementation
**********************************************************************/

package org.eclipse.hyades.execution.core;

/** 
 * 
 * <p>IExecutionComponent is the base class for all of the components in the
 * runtime system.  IExecutionComponent's are structured as a tree with
 * the specialized ISession is the root of the tree on the workbench side.</p>
 * 
 * <p>IExecutionComponent's are uniquely identified at runtime with an ID.
 * They are also named and typed but names and types are not guaranteed
 * to be unique.  Version support is provided as well.</p>
 * 
 * <p>IExecutionComponent's have at least four states.  They can be inactive,
 * ready, suspended, or dead.  <ul><li>Inactive IExecutionComponent's are
 * models of components that have not been created yet.</li> <li>Dead
 * IExecutionComponent's are those that will never be ready again.</li>
 * <li>Ready IExecutionComponent's are those that can produce or consume
 * IMessage's and IRuntimeVariable's.</li> <li>Suspended IExecutionComponent's
 * are are those that have been created and are expected to become ready
 * at some point in the future.</li></ul></p>
 * @version 0.0.2
 */
public interface IExecutionComponent {
	
	/** 
	 * All IExecutionComponent instances must have an immutable
	 * name.  Names are user friendly identifiers.
	 * 
	 * @return the name of the IExecutionComponent
	 */
	public String getName();
	
	/** 
	 * All IExecutionComponent instances must have an immutable
	 * ID. Id's are globaly unique values that identify this unique
	 * IExecutionComponent runtime instance.
	 * 
	 * @return The ID of the IExecutionComponent
	 */
	public String getId();
	
	/** 
	 * All IExecutionComponent instances must have an immutable
	 * type. The type field is intended to allow you to identify groups
	 * of IExecutionComponent instances as having the same behaviour.
	 * 
	 * @return the type of the IExecutionComponent
	 */
	public String getType();
	
	/** 
	 * Retrieve the version.
	 */
	public String getVersion();
	
	/** 
	 * Get the container where this IExecutionComponent
	 * is hosted.  The typical heirarchy is a follows:
	 * ISession->IExecutor->IRemoteHyadesComponent
	 * It is possible, however, to have any level of nesting here.
	 * The ISession is always the root.
	 */
	IExecutionComponent getParent();
	
	/** 
	 * Get the first-level children that reside under this IExecutionComponent
	 * The typical heirarchy is a follows:
	 * ISession->IExecutor->IRemoteHyadesComponent
	 * It is possible, however, to have any level of nesting here.
	 * The ISession is always the root.
	 */
	IExecutionComponent[] getChildren();
	
	
	/** 
	 * Retrieve all of the first-level children of a specific type.
	 */
	IExecutionComponent[] getChildren(String  type);

	/**
	 * Retrieve all of the first-level children of a specific type in the specified state.
	 */
	IExecutionComponent[] getChildren(String type, int state);
	
	/** 
	 * Retrieve a child of this IExecutionComponent based upon its
	 * ID.
	 */
	public IExecutionComponent getChildById(String id);
	
	/** 
	 * Retrieve a child of this IExecutionComponent based upon its
	 * name.
	 * RKD:  This may not be the best idea due to name collision
	 * issues.  probably want to stick to just ID's.
	 * 
	 * @param name 
	 */
	public IExecutionComponent getChildByName(String name);
	
	/** 
	 * Create an child
	 * 
	 * @param id ID of the child IExecutionComponent
	 * @param name Name of the child IExecutionComponent
	 * @param type Type of the child IExecutionComponent
	 * @return the child IExecutionComponent
	 */
	public IExecutionComponent createChild(String id, String name, String type);

	/** 
	 * Add a child
	 * 
	 * @param id ID of the child IExecutionComponent
	 * @param name Name of the child IExecutionComponent
	 * @param type Type of the child IExecutionComponent
	 */
	public void addChild(String id, String name, String type);

	/** 
	 * Get the node on which this IExecutionComponent resides.
	 * 
	 * @return the node on which this IExecutionComponent resides
	 */
	public INode getNode();
			
	/**
	 * Retrieve the current state of this component
	 */
	public IExecutionComponentState getState();
		
	/**
	 * Adds an IExecCompStateChangeListener to this IExecutionComponent
	 * @param listener the listener to be informed of state change events.
	 */
	public void addExecCompStateChangeListener(IExecCompStateChangeListener listener);

	public void removeExecCompStateChangeListener(IExecCompStateChangeListener listener);
	
	public void fireStateChange(IExecCompStateChangeEvent event);
}
