/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.core;

import java.io.*;

/**
 * 
 * Description:
 *		This type represents a data channel between two processes.
 * 		The user process can either fill the data channel or flush the
 * 		data channel.  They cannot do both.
 * 		The filler of the data channel provides an
 * 		InputStream that the data channel reads from to fill the channel.
 * 		The flusher of the data channel provides an OutputStream that the 
 * 		data channel writes the data on the channel to.
 * 
 * Creation date: Apr 9, 2003
 * 
 */

public interface IDataChannel  {

/**
 * Sets the name of the data channel, otherwise a default name is used.
 * @param name java.lang.String
 */
public void setName(String name);

/**
 * Gets the name of the data channel.
 * @return java.lang.String
 */
public String getName();

/**
 * Sets the size of the data channel, otherwise a default size is used.
 * @param size int
 */
public void setSize(int size);

/**
 * Gets the size of the data channel.
 * @return int
 */
public int getSize();

/**
 * Creates the data channel.
 * @throws DataChannelCreationException
 */

public void create() throws DataChannelCreationException;

/**
 * Destroys the data channel.
 */

public void destroy();

/**
 * Connects to the data channel.
 * @throws DataChannelConnectionException
 */

public void connect() throws DataChannelConnectionException;

/**
 * Disconnects from the data channel.
 */

public void disconnect();

/**
 * Sets this data channel as flushable.
 * @throws InvalidDataChannelAccessException
 */

public void setFlushable() throws InvalidDataChannelAccessException;

/**
 * Tests whether the application can flush this data channel.
 * @return boolean
 */

public boolean isFlushable();

/**
 * Sets this data channel as fillable.
 * @throws InvalidDataChannelAccessException
 */

public void setFillable() throws InvalidDataChannelAccessException;;

/**
 * Tests whether the application can fill this data channel.
 * @return boolean
 */

public boolean isFillable();



/**
 * Returns an output stream that can be used to write to the data channel.
 * This is called by the filler of the data channel.
 * @return java.io.OutputStream
 * @throws InvalidDataChannelAccessException
 * @throws DataChannelNotInitializedException
 */
public OutputStream getOutputStream()
				throws InvalidDataChannelAccessException, DataChannelNotInitializedException;


/**
 * Returns an input stream that can be used to read from the data channel.
 * This is called by the flusher of the data channel.
 * @return java.io.InputStream
 * @throws InvalidDataChannelAccessException
 * @throws DataChannelNotInitializedException
 */
public InputStream getInputStream()
				throws InvalidDataChannelAccessException, DataChannelNotInitializedException;


}
