/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.local.control;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.security.Principal;
import org.eclipse.hyades.internal.execution.security.DuplicateUserException;
import org.eclipse.hyades.internal.execution.security.User;

/* COMPLETE and UNTESTED */

/**
 * A factory for creating Nodes.  Nodes can be specified by either name
 * or InetAddress.  Each Node is treated as a singleton and
 * attempts to create a Node that already exists will return
 * the same Node instance that already existed.
 */
public class NodeFactory {
	private static final int INITIAL_SERVERS=16;
	private static Node[] _servers=new Node[INITIAL_SERVERS];
	
	static {
		try {
			_servers[0]=createNode(InetAddress.getLocalHost());
		}
		catch(UnknownHostException e) {
			/* We ignore this here and throw the error
			   when getLocalHost is called later.
			*/
		}
	}
	
	/**
	 * Create a Node for the suplied hostname.  Nodes are
	 * are intended to be singletons in this application so
	 * if the Node exists it will be returned.
	 */
	public static Node createNode(String hostname) throws UnknownHostException{
		try {
			return createNode(hostname, null);
		}
		catch(DuplicateUserException e) {
			return containsNode(hostname, null);
		}
		
	}
	
	
	public static Node createNode(String hostname, Principal principal) throws UnknownHostException, DuplicateUserException {
		Node result=containsNode(hostname, principal);
		if(result==null) {
			InetAddress addr=InetAddress.getByName(hostname);
			return addNode(hostname, addr, principal);
		}
		throw new DuplicateUserException();
		
	}
	
	
	
	/**
	 * Create a Node for the suplied InetAddress.  Nodes are
	 * are intended to be singletons in this application so
	 * if the Node exists it will be returned.
	 */
	public static Node createNode(InetAddress address) throws UnknownHostException {
		return createNode(address, null);
	}
	
	public static Node createNode(InetAddress address, Principal principal) throws UnknownHostException {
		String hostname=address.getHostName();
		Node result=containsNode(hostname, principal);
		if(result==null) {
			result=addNode(hostname, address, principal);
		}
		return result;
	}
	
	/**
	 * Returns the node that represents the local host.
	 * @deprecated - use getLocalHost(User user)
	 */
	public static Node getLocalHost() throws UnknownHostException {
		/* The localhost is always in the first slot of the servers table */
		if(_servers[0]==null) {
			throw new UnknownHostException();
		}
		return _servers[0];	
	}
	
	public static Node getLocalHost(Principal principal) throws UnknownHostException {
		/* The localhost is always in the first slot of the servers table */
		if(_servers[0]==null) {
			throw new UnknownHostException();
		}
		
		String hostname= _servers[0].getName();
		
		
		
		return null;	
	}
	
	private static Node addNode(String name, InetAddress addr, Principal principal) {
		
		if(name.equals("localhost")) {
			try {
				name=InetAddress.getLocalHost().getHostName();
				InetAddress[] addrs=InetAddress.getAllByName(name);
				addr=addrs[0];
			}
			catch(UnknownHostException e) {
				/* We can ignore this */
			}
		}
	
		synchronized(_servers) {
			/* Look for the first empty slot */
			for(int i=0; i<_servers.length; i++) {
				if(_servers[i]==null) {
					_servers[i]=new NodeImpl(name, addr);
					
					/* Currently we know how to deal with applications and users */
					if(principal instanceof User) {
						((NodeImpl)_servers[i]).setUser((User)principal);
					}
					else if(principal instanceof Application) {
						((NodeImpl)_servers[i]).setApplication((Application)principal);
					}
					return _servers[i];
				}
			}
			/* The table is full, increase it's size. */
			Node[] oldEntries=_servers;
			_servers= new Node[oldEntries.length+1];
			for(int i=0; i<oldEntries.length; i++) {
				_servers[i]=oldEntries[i];
			}
			_servers[oldEntries.length]=new NodeImpl(name, addr);
			return _servers[oldEntries.length];
		}
	}
	
	/**
	 * Searches the Node list based upon the hostname.
	 * @return the Node if it exists, null otherwise.
	 */
	private static Node containsNode(String name) {
		return containsNode(name, null);
	}
	
	private static Node containsNode(String name, Principal principal) {
		synchronized(_servers) {
			/* If this is "localhost" try and resolve its real name first */
			if(name.equals("localhost")) {
				try {
					name=InetAddress.getLocalHost().getHostName();
				}
				catch(UnknownHostException e) {
					/* We can ignore this */
				}
			}
			InetAddress[] addresses=null;
			try {
				addresses=InetAddress.getAllByName(name);
				
				addresses=InetAddress.getAllByName(addresses[0].getHostName());
			}
			catch(UnknownHostException e) {
				return null;
			}
			
			for(int i=0; i<_servers.length; i++) {
				if(_servers[i]!=null) {
					for(int k=0; k<addresses.length; k++) {
						InetAddress temp=_servers[i].getInetAddress();
						if(addresses[k].equals(temp)) {
							if(principal==null) { 
								return _servers[i];
							}
							else if(principal instanceof User && _servers[i].getUser()!=null) {
								if(principal.getName().equals(_servers[i].getUser().getName())) {
									return _servers[i];
								}	
							}
							else if(principal instanceof Application && _servers[i].getApplication()!=null) {
								if(principal.getName().equals(_servers[i].getApplication().getName())) {
									return _servers[i];
								}	
							}
						}
					}
				}
			}
		}
		return null;
	}
	
	/**
	 * Searches the Node list based upon the InetAddress.
	 * @return the Node if it exists, null otherwise.
	 */
	public static Node getNode(InetAddress addr) {
		return getNode(addr, null);
	}
	
	public static Node getNode(InetAddress addr, Principal principal) {
		/* Resolve the name and delegate */
		return containsNode(addr.getHostName(), principal);
	}
	
	public static Node getNode(String name, Principal principal) throws UnknownHostException{
		/* Resolve the name and delegate */
		return getNode(InetAddress.getByName(name), principal);
	}

}

