package org.iiter.continuum.protos.agent;

import java.io.StringWriter;
import java.lang.instrument.ClassFileTransformer;
import java.lang.instrument.IllegalClassFormatException;
import java.net.URL;
import java.security.ProtectionDomain;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Transformer for information on classloader
 */
public class LogClassTransformer implements ClassFileTransformer {
	
	private Set<AdaptorKey> classloaders = Collections.synchronizedSet(new HashSet<AdaptorKey>());
	
	private static final String AOP_XML = "META-INF/aop.xml;META-INF/aop-ajc.xml;org/aspectj/aop.xml";
	
	// Logger SLF4J
	private static final Logger logger = LoggerFactory.getLogger(LogClassTransformer.class);
	
	@Override
	public byte[] transform(ClassLoader loader, String className, Class<?> classBeingRedefined, ProtectionDomain protectionDomain, byte[] classfileBuffer)
			throws IllegalClassFormatException {
		AdaptorKey key = new AdaptorKey(loader);
		if (!classloaders.contains(key)) {
			logger.info("Searching in classloader {}...", loader.getClass().getName());
			StringTokenizer st = new StringTokenizer(AOP_XML, ";");
			while (st.hasMoreTokens()) {
				String nextDefinition = st.nextToken();
				URL url = loader.getResource(nextDefinition);
				if (url == null) {
					logger.info("\t{} not found in classloader {}", nextDefinition, loader.getClass().getName());
				} else {
					logger.info("{} found in classloader {}: {}", nextDefinition, loader.getClass().getName(), url.toString());
				}
			}
			classloaders.add(key);
			writeHierarchy();
		}
		return classfileBuffer;
	}

	protected synchronized void writeHierarchy() {
		logger.info("Writing {} classloader hierarchy", classloaders.size());
		HierarchyNode<AdaptorKey> root = new HierarchyNode<AdaptorKey>(null, null);
		for (AdaptorKey key : classloaders) {
			List<AdaptorKey> cls = new ArrayList<AdaptorKey>();
			cls.add(key);
			
			// Récupération de la liste des classloaders parents
			ClassLoader cl = key.getClassLoader();
			try {
				ClassLoader parentCl = cl.getParent();
				while (parentCl != null) {
					cls.add(new AdaptorKey(parentCl));
					parentCl = parentCl.getParent();
				}
			} catch (Exception e) {
				logger.error("Error looking for classloader's parent", e);
			}
			
			// Ajout à la hiérarchie
			AdaptorKey curKey;
			int nbCls = cls.size();
			HierarchyNode<AdaptorKey> curNode = root;
			while (nbCls-- > 0) {
				curKey = cls.get(nbCls);
				HierarchyNode<AdaptorKey> nextNode = curNode.getChild(curKey);
				if (nextNode == null) {
					curNode = curNode.addChild(curKey);
				} else {
					curNode = nextNode;
				}
			}
		}
		try {
			StringWriter writer = new StringWriter();
			root.write(writer);
			logger.info(writer.toString());
		} catch (Exception e) {
			logger.error("Error writing classloader hierarchy", e);
		}
	}

}
