/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * This file is part of the compiler and core tools for the AspectJ(tm)
 * programming language; see http://aspectj.org
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * either http://www.mozilla.org/MPL/ or http://aspectj.org/MPL/.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is AspectJ.
 *
 * The Initial Developer of the Original Code is Xerox Corporation. Portions
 * created by Xerox Corporation are Copyright (C) 1999-2001 Xerox Corporation.
 * All Rights Reserved.
 *
 * Contributor(s):
 */

package org.aspectj.tools.ajdoc;

import java.io.File;
import java.io.FileOutputStream;
import java.io.BufferedWriter;
import java.io.PrintWriter;
import java.io.IOException;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.StringReader;
import java.io.FilenameFilter;

import java.util.*;

import org.aspectj.tools.ide.SymbolManager;
import org.aspectj.tools.ide.Declaration;
import org.aspectj.tools.ide.SourceLine;

public class Main implements Config {

    static SymbolManager symbolManager = null;

    /** Command line options. */
    static Vector options = new Vector();

    /** Options to pass to ajc. */
    static Vector ajcOptions = new Vector();

    /** All of the files to be processed by ajdoc. */
    static Vector filenames = new Vector();

    /** List of files to pass to javadoc. */
    static Vector fileList= new Vector();

    /** List of packages to pass to javadoc. */
    static Vector packageList = new Vector();

    /** Default to package visiblity. */
    static String docModifier = "package";

    static Vector sourcepath = new Vector();

    static boolean verboseMode = false;
    static boolean packageMode = false;
    static boolean authorStandardDocletSwitch = false;
    static boolean versionStandardDocletSwitch = false;
    static File    rootDir       = null;
    static Hashtable declIDTable   = new Hashtable();
    static String  docDir          = ".";

    public static void clearState() {
        symbolManager = null;
        options = new Vector();
        ajcOptions = new Vector();
        filenames = new Vector();
        fileList= new Vector();
        packageList = new Vector();
        docModifier = "package";
        sourcepath = new Vector();
        verboseMode = false;
        packageMode = false;
        rootDir       = null;
        declIDTable   = new Hashtable();
        docDir          = ".";
    }

    public static void main(String[] args) {

        // STEP 1: parse the command line and do other global setup
        sourcepath.addElement("."); // add the current directory to the classapth
        parseCommandLine(args);
        rootDir = getRootDir();
        symbolManager = SymbolManager.getSymbolManager();
        File[] inputFiles      = new File[filenames.size()];
        File[] signatureFiles  = new File[filenames.size()];
        try {
            // create the workingdir if it doesn't exist
            if ( !(new File( Config.WORKING_DIR ).isDirectory()) ) {
                File dir = new File( Config.WORKING_DIR );
                dir.mkdir();
                dir.deleteOnExit();
            }

            for (int i = 0; i < filenames.size(); i++) {
                inputFiles[i]     = findFile((String)filenames.elementAt(i));
                //signatureFiles[i] = createSignatureFile(inputFiles[i]);
            }

            // PHASE 0: call ajc
            ajcOptions.addElement( "-preprocess" );
            String[] argsToCompiler = new String[ajcOptions.size() + inputFiles.length];
            int i = 0;
            for ( ; i < ajcOptions.size(); i++ ) {
                argsToCompiler[i] = (String)ajcOptions.elementAt(i);
                //System.out.println(">>> ajc: " + argsToCompiler[i]);
            }
            for ( int j = 0; j < inputFiles.length; j++) {
                argsToCompiler[i] = inputFiles[j].getAbsolutePath();
                //System.out.println(">> file to ajc: " + inputFiles[j].getAbsolutePath());
                i++;
            }

            System.out.println( "> calling ajc..." );

            try {
                (new org.aspectj.tools.ajc.Main() {
                    public void exit(int status) {
                    if (status != 0) {
                    System.out.println( "> compile failed, exiting ajdoc" );
                    System.exit( -1 );
                    }
                    }
                    }).realMain(argsToCompiler);
            }
            catch ( SecurityException se ) {
                System.exit( -1 );
            }
/*
            for (int ii = 0; ii < inputFiles.length; ii++) {
                String tempFP = inputFiles[ii].getAbsolutePath();
                tempFP = tempFP.substring(0, tempFP.length()-4);
                tempFP += "ajsym";
                System.out.println( ">> checking: " + tempFP);
                File tempF = new File(tempFP);
                if ( !tempF.exists() ) System.out.println( ">>> doesn't exist!" );
            }
*/
            for (int ii = 0; ii < filenames.size(); ii++) {
                signatureFiles[ii] = createSignatureFile(inputFiles[ii]);
            }

            // PHASE 1: generate Signature files (Java with DeclIDs and no bodies).
            System.out.println( "> building signature files..." );
            Phase1.doFiles(declIDTable, symbolManager, inputFiles, signatureFiles);

            // PHASE 2: let Javadoc generate HTML (with DeclIDs)
            System.out.println( "> calling javadoc..." );
            String[] javadocargs = null;
            if ( packageMode ) {
                int numExtraArgs = 2;
                if (authorStandardDocletSwitch) numExtraArgs++;
                if (versionStandardDocletSwitch) numExtraArgs++;
                javadocargs = new String[numExtraArgs + options.size() + packageList.size() +
                                         fileList.size() ];
                javadocargs[0] = "-sourcepath";
                javadocargs[1] = Config.WORKING_DIR;
                int argIndex = 2;
                if (authorStandardDocletSwitch) {
                    javadocargs[argIndex] = "-author";
                    argIndex++;
                }
                if (versionStandardDocletSwitch) {
                    javadocargs[argIndex] = "-version";
                }
                //javadocargs[1] = getSourcepathAsString();
                for (int k = 0; k < options.size(); k++) {
                    javadocargs[numExtraArgs+k] = (String)options.elementAt(k);
                }
                for (int k = 0; k < packageList.size(); k++) {
                    javadocargs[numExtraArgs+options.size() + k] = (String)packageList.elementAt(k);
                }
                for (int k = 0; k < fileList.size(); k++) {
                    javadocargs[numExtraArgs+options.size() + packageList.size() + k] = (String)fileList.elementAt(k);
                }
            }
            else {
                javadocargs = new String[options.size() + signatureFiles.length];
                for (int k = 0; k < options.size(); k++) {
                    javadocargs[k] = (String)options.elementAt(k);
                }
                for (int k = 0; k < signatureFiles.length; k++) {
                    javadocargs[options.size() + k] = signatureFiles[k].getCanonicalPath();
                }
            }
            Phase2.callJavadoc(javadocargs);

            //for ( int o = 0; o < inputFiles.length; o++ ) {
            //    System.out.println( "file: " + inputFiles[o] );
            //}

            // PHASE 3: add AspectDoc specific stuff to the HTML (and remove the DeclIDS).
            /** We start with the known HTML files (the ones that correspond directly to the
            * input files.)  As we go along, we may learn that Javadoc split one .java file
            * into multiple .html files to handle inner classes or local classes.  The html
            * file decorator picks that up.
            */
            System.out.println( "> Decorating html files..." );
            Phase3.decorateHTMLFromInputFiles(declIDTable,
                                              rootDir,
                                              symbolManager,
                                              inputFiles,
                                              docModifier);
            removeDeclIDsFromFile("index-all.html");
            removeDeclIDsFromFile("serialized-form.html");
            for (int p = 0; p < packageList.size(); p++) {
                removeDeclIDsFromFile(((String)packageList.elementAt(p)).replace('.','/') +
                                       Config.DIR_SEP_CHAR +
                                       "package-summary.html" );
            }
        } catch (Throwable e) {
            handleInternalError(e);
            exit(-2);
        }
    }

    static void removeDeclIDsFromFile(String filename) {
        // Remove the decl ids from "index-all.html"
        File indexFile = new File( docDir + Config.DIR_SEP_CHAR + filename );
        try {
        if ( indexFile.exists() ) {
            BufferedReader indexFileReader = new BufferedReader( new FileReader( indexFile ) );
            String indexFileBuffer = "";
            String line = indexFileReader.readLine();
            while ( line != null ) {
              int indexStart = line.indexOf( Config.DECL_ID_STRING );
              int indexEnd   = line.indexOf( Config.DECL_ID_TERMINATOR );
              if ( indexStart != -1 && indexEnd != -1 ) {
                line = line.substring( 0, indexStart ) +
                       line.substring( indexEnd+Config.DECL_ID_TERMINATOR.length() );
              }
              indexFileBuffer += line;
              line = indexFileReader.readLine();
            }
            FileOutputStream fos = new FileOutputStream( indexFile );
            fos.write( indexFileBuffer.getBytes() );
        }
        }
        catch (IOException ioe) {
              // be siltent
        }
    }

    static void callAJC( String[] ajcargs ) {
        final SecurityManager defaultSecurityManager = System.getSecurityManager();

        System.setSecurityManager( new SecurityManager() {
            public void checkExit(int status) {
            System.setSecurityManager( defaultSecurityManager );
            //throw new SecurityException( "status code: " + status );
            //System.exit( -1 );
            }
            public void checkPermission( java.security.Permission permission ) {
            if ( defaultSecurityManager  != null )
            defaultSecurityManager.checkPermission( permission );
            }
            public void checkPermission( java.security.Permission permission,
            Object context ) {
            if ( defaultSecurityManager  != null )
            defaultSecurityManager.checkPermission( permission, context );
            }
            } );

        try {
            org.aspectj.tools.ajc.Main.main( ajcargs );
        }
        catch ( SecurityException se ) {
            // Do nothing since we expect it to be thrown
            //System.out.println( ">> se: " + se.getMessage() );
        } catch ( IOException ioe ) {
            System.out.println( ">> io error: " + ioe.getMessage() );
        }
        // Set the security manager back
        System.setSecurityManager( defaultSecurityManager );
    }


    /**
     * If the file doesn't exist on the specified path look for it in all the other
     * paths specified by the "sourcepath" option.
     */
    static File findFile( String filename ) throws IOException {

        return new File( filename );
        /*
        File file = new File(filename);
        if (file.exists()) {
            return file;
        }
        else {
            for ( int i = 0; i < sourcePath.size(); i++ ) {
                File currPath = new File((String)sourcePath.elementAt(i));
                File currFile = new File(currPath + "/" + filename); // !!!
                if ( file.exists()) {
                    return file;
                }
            }
        }
        throw new IOException("couldn't find source file: " + filename);
        */
    }

    static Vector getSourcePath() {
        Vector  sourcePath = new Vector();
        boolean found      = false;
        for ( int i = 0; i < options.size(); i++ ) {
            String currOption = (String)options.elementAt(i);
            if (found && !currOption.startsWith("-")) {
                sourcePath.add(currOption);
            }
            if (currOption.equals("-sourcepath")) {
                found = true;
            }
        }
        return sourcePath;
    }

    static File getRootDir() {
        File rootDir = new File( "." );
        for ( int i = 0; i < options.size(); i++ ) {
            if ( ((String)options.elementAt(i)).equals( "-d" ) ) {
                rootDir = new File((String)options.elementAt(i+1));
                if ( !rootDir.exists() ) {
                    System.out.println( "Destination directory not found: " +
                                        (String)options.elementAt(i+1) );
                    System.exit( -1 );
                }
            }
        }
        return rootDir;
    }

    static File createSignatureFile(File inputFile) throws IOException {
        Declaration[] decls = symbolManager.getDeclarations(inputFile.getAbsolutePath());
        Declaration decl = null;
        String packageName = null;
        if ( decls != null && decls[0] != null ) {
            decl = decls[0];
            packageName = decl.getPackageName();
        }
        String filename    = "";
        if ( packageName != null ) {
            //System.err.println(">> creating: " + packageName);
            String pathName =  Config.WORKING_DIR + '/' + packageName.replace('.', '/');
            //System.err.println(">> pathName: " + pathName);
            File packageDir = new File(pathName);
            if ( !packageDir.exists() ) {
                packageDir.mkdirs();
                //packageDir.deleteOnExit();
            }
            //verifyPackageDirExists(packageName, null);
            packageName = packageName.replace( '.','/' ); // !!!
            filename = Config.WORKING_DIR + Config.DIR_SEP_CHAR + packageName +
                       Config.DIR_SEP_CHAR + inputFile.getName();
        }
        else {
            filename = Config.WORKING_DIR + Config.DIR_SEP_CHAR + inputFile.getName();
        }
        File signatureFile = new File( filename );
        //signatureFile.deleteOnExit();
        return signatureFile;
    }


    static void verifyPackageDirExists( String packageName, String offset ) {
        System.err.println(">>> name: " + packageName + ", offset: " + offset);
        if ( packageName.indexOf( "." ) != -1 ) {
            File tempFile = new File("c:/aspectj-test/d1/d2/d3");
            tempFile.mkdirs();
            String currPkgDir = packageName.substring( 0, packageName.indexOf( "." ) );
            String remainingPkg = packageName.substring( packageName.indexOf( "." )+1 );
            String filePath = null;
            if ( offset != null ) {
                filePath = Config.WORKING_DIR + Config.DIR_SEP_CHAR +
                           offset + Config.DIR_SEP_CHAR + currPkgDir ;
            }
            else {
                filePath = Config.WORKING_DIR + Config.DIR_SEP_CHAR + currPkgDir;
            }
            File packageDir = new File( filePath );
            if ( !packageDir.exists() ) {
                packageDir.mkdir();
                packageDir.deleteOnExit();
            }
            if ( remainingPkg != "" ) {
                verifyPackageDirExists( remainingPkg, currPkgDir );
            }
        }
        else {
            String filePath = null;
            if ( offset != null ) {
                filePath = Config.WORKING_DIR + Config.DIR_SEP_CHAR + offset + Config.DIR_SEP_CHAR + packageName;
            }
            else {
                filePath = Config.WORKING_DIR + Config.DIR_SEP_CHAR + packageName;
            }
            File packageDir = new File( filePath );
            if ( !packageDir.exists() ) {
                packageDir.mkdir();
                packageDir.deleteOnExit();
            }
        }
    }

    static void parseCommandLine(String[] args) {
        if (args.length == 0) {
            displayHelpAndExit( null );
        }
        List vargs = new LinkedList(Arrays.asList(args));

        parseArgs(vargs, new File( "." ));  // !!!

        if (filenames.size() == 0) {
            displayHelpAndExit( "ajdoc: No packages or classes specified" );
        }
    }

    static void setSourcepath(String arg) {
           sourcepath.clear();
           arg = arg + ";"; // makes things easier for ourselves
           StringTokenizer tokenizer = new StringTokenizer(arg, ";");
           while (tokenizer.hasMoreElements()) {
                 sourcepath.addElement(tokenizer.nextElement());
           }
    }

    static String getSourcepathAsString() {
       String cPath = "";
       for (int i = 0; i < sourcepath.size(); i++) {
           cPath += (String)sourcepath.elementAt(i) + Config.DIR_SEP_CHAR + Config.WORKING_DIR;
           if (i != sourcepath.size()-1) {
              cPath += ";";
           }
       }
       return cPath;
    }

    static void parseArgs(List vargs, File currentWorkingDir) {
        boolean addNextAsOption     = false;
        boolean addNextAsArgFile    = false;
        boolean addNextToAJCOptions = false;
        boolean addNextAsDocDir     = false;
        boolean addNextAsClasspath  = false;
        boolean ignoreArg           = false;  // used for discrepancy betwen class/sourcepath in ajc/javadoc
        boolean addNextAsSourcePath = false;
        if ( vargs.size() == 0 ) {
            displayHelpAndExit( null );
        }
        for (int i = 0; i < vargs.size() ; i++) {
            String arg = (String)vargs.get(i);
            ignoreArg = false;
            if ( addNextToAJCOptions ) {
                ajcOptions.addElement( arg );
                addNextToAJCOptions = false;
            }
            if ( addNextAsDocDir ) {
                docDir = arg;
                addNextAsDocDir = false;
            }
            if ( addNextAsClasspath ) {
                addNextAsClasspath = false;
            }
            if ( addNextAsSourcePath ) {
                setSourcepath( arg );
                addNextAsSourcePath = false;
                ignoreArg = true;
            }
            if ( arg.startsWith("@") ) {
                expandAtSignFile(arg.substring(1), currentWorkingDir);
            }
            else if ( arg.equals( "-argfile" ) ) {
                addNextAsArgFile = true;
            }
            else if ( addNextAsArgFile ) {
                expandAtSignFile(arg, currentWorkingDir);
                addNextAsArgFile = false;
            }
            else if (arg.equals("-d") ) {
                addNextAsOption = true;
                options.addElement(arg);
                addNextAsDocDir = true;
            }
            else if ( arg.equals( "-bootclasspath" ) ) {
                addNextAsOption = true;
                addNextToAJCOptions = true;
                options.addElement( arg );
                ajcOptions.addElement( arg );
            }
            else if ( arg.equals( "-classpath" ) ) {
                addNextAsOption = true;
                addNextToAJCOptions = true;
                addNextAsClasspath = true;
                options.addElement( arg );
                ajcOptions.addElement( arg );
            }
            else if ( arg.equals( "-sourcepath" ) ) {
                addNextAsSourcePath = true;
                //options.addElement( arg );
                //ajcOptions.addElement( arg );
            }
            else if (arg.startsWith("-") || addNextAsOption) {
                if ( arg.equals( "-private" ) ) {
                    docModifier = "private";
                }
                else if ( arg.equals( "-package" ) ) {
                    docModifier = "package";
                }
                else if ( arg.equals( "-protected" ) ) {
                    docModifier = "protected";
                }
                else if ( arg.equals( "-public" ) ) {
                    docModifier = "public";
                }
                else if ( arg.equals( "-verbose" ) ) {
                    verboseMode = true;
                }
                else if ( arg.equals( "-author" ) ) {
                    authorStandardDocletSwitch = true;
                }
                else if ( arg.equals( "-version" ) ) {
                    versionStandardDocletSwitch = true;
                }
                else if ( arg.equals( "-v" ) ) {
                    printVersion();
                    exit(0);
                }
                else if ( arg.equals( "-help" ) ) {
                    displayHelpAndExit( null );
                }
                else if ( arg.equals( "-doclet" ) || arg.equals( "-docletpath" ) ) {
                    System.out.println( "The doclet and docletpath options are not currently supported    \n" +
                                        "since ajdoc makes assumptions about the behavior of the standard \n" +
                                        "doclet. If you would find this option useful please email us at: \n" +
                                        "                                                                 \n" +
                                        "       asupport@aspectj.org                            \n" +
                                        "                                                                 \n" );
                    exit(0);
                }
                else if ( addNextAsOption ) {
                    // just pass through
                }
                else {
                    displayHelpAndExit( null );
                }
                options.addElement(arg);
                addNextAsOption = false;
            }
            else {
                // check if this is a file or a package
                if ( arg.indexOf( ".java" ) == arg.length() - 5 ||
                     arg.indexOf( ".lst" ) == arg.length() - 4 &&
                     arg != null ) {
                    File f = new File(arg);
                    if (f.isAbsolute()) {
                         filenames.addElement(arg);
                    }
                    else {
                         filenames.addElement( currentWorkingDir + Config.DIR_SEP_CHAR + arg );
                    }
                    fileList.addElement( arg );
                }

                // PACKAGE MODE STUFF
                else if (!ignoreArg) {
                    packageMode = true;
                    packageList.addElement( arg );
                    arg = arg.replace( '.', '/' );  // !!!

                    // do this for every item in the classpath
                    for ( int c = 0; c < sourcepath.size(); c++ ) {
                      String path = (String)sourcepath.elementAt(c) + Config.DIR_SEP_CHAR + arg;
                      File pkg = new File(path);
                      if ( pkg.isDirectory() ) {
                          String[] files = pkg.list( new FilenameFilter() {
                              public boolean accept( File dir, String name )  {
                                  int index1 = name.lastIndexOf( "." );
                                  int index2 = name.length();
                                  if ( (index1 >= 0 && index2 >= 0 ) &&
                                        (name.substring(index1, index2).equals( ".java" ) ) ) {
                                      return true;
                                  }
                                  else  {
                                      return false;
                                  }
                              }
                             } );
                          for ( int j = 0; j < files.length; j++ ) {
                              filenames.addElement( (String)sourcepath.elementAt(c) +
                                                    Config.DIR_SEP_CHAR +
                                                    arg + Config.DIR_SEP_CHAR + files[j] );
                          }
                      }
                      else if (c == sourcepath.size() ) { // last element on classpath
                          System.out.println( "ajdoc: No package, class, or source file " +
                                              "found named " + arg + "." );
                      }
                      else {
                           // didn't find it on that element of the classpath but that's ok
                      }
                    }
                }
            }
        }
        // set the default visibility as an option to javadoc option
        if ( !options.contains( "-private" ) &&
             !options.contains( "-package" ) &&
             !options.contains( "-protected" ) &&
             !options.contains( "-public" ) ) {
            options.addElement( "-package" );
        }

    }

    static void expandAtSignFile(String filename, File currentWorkingDir) {
        List result = new LinkedList();

        File atFile = qualifiedFile(filename, currentWorkingDir);
        String atFileParent = atFile.getParent();
        File myWorkingDir = null;
        if (atFileParent != null) myWorkingDir = new File(atFileParent);

        try {
            BufferedReader stream = new BufferedReader(new FileReader(atFile));
            String line = null;
            while ( (line = stream.readLine()) != null) {
                // strip out any comments of the form # to end of line
                int commentStart = line.indexOf("//");
                if (commentStart != -1) {
                    line = line.substring(0, commentStart);
                }

                // remove extra whitespace that might have crept in
                line = line.trim();
                // ignore blank lines
                if (line.length() == 0) continue;
                result.add(line);
            }
        } catch (IOException e) {
            System.err.println("Error while reading the @ file " + atFile.getPath() + ".\n"
                               + e);
            System.exit( -1 );
        }

        parseArgs(result, myWorkingDir);
    }

    static File qualifiedFile(String name, File currentWorkingDir) {
        name = name.replace('/', File.separatorChar);
        File file = new File(name);
        if (!file.isAbsolute() && currentWorkingDir != null) {
            file = new File(currentWorkingDir, name);
        }
        return file;
    }


    static void displayHelpAndExit(String message) {
        if (message != null) System.err.println(message);
        System.err.println();
        System.err.println(Config.USAGE);
        exit(0);
    }

    static protected void exit(int value) {
        System.out.flush();
        System.err.flush();
        System.exit(value);
    }

    /* This section of code handles errors that occur during compilation */
    static final String internalErrorMessage =
                                              "Please copy the following text into an email message and send it,\n" +
                                              "along with any additional information you can add to:            \n" +
                                              "                                                                 \n" +
                                              "       support@aspectj.org                           \n" +
                                              "                                                                 \n";

    static public void handleInternalError(Throwable uncaughtThrowable) {
        System.err.println("An internal error occured in ajdoc");
        System.err.println(internalErrorMessage);
        System.err.println(uncaughtThrowable.toString());
        uncaughtThrowable.printStackTrace();
        System.err.println();
    }

    static void printVersion() {
        System.out.println( Config.VERSION );
    }

}


