// Copyright: Apache2 license
package test.teneo;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.Properties;

import junit.framework.Assert;
import junit.framework.TestCase;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.impl.EStringToStringMapEntryImpl;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.teneo.hibernate.HbDataStore;
import org.eclipse.emf.teneo.hibernate.HbHelper;
import org.eclipse.emf.teneo.hibernate.resource.HibernateResource;
import org.hibernate.SessionException;
import org.hibernate.cfg.Environment;
import org.hibernate.dialect.HSQLDialect;

import test.teneo.map.testmap.TableWithMap;
import test.teneo.map.testmap.TestmapFactory;
import test.teneo.map.testmap.TestmapPackage;

public class MapTests extends TestCase {
	private static final String DBNAME="MapTestsDB";
	
    private static Properties getHibernateProperties(String dbName) {
        final Properties props = new Properties(); 
        props.setProperty(Environment.DRIVER, "org.hsqldb.jdbcDriver");
        props.setProperty(Environment.USER, "sa");
        props.setProperty(Environment.URL, "jdbc:hsqldb:file:" + dbName + ";shutdown=true");
        props.setProperty(Environment.PASS, "");
        props.setProperty(Environment.DIALECT, HSQLDialect.class.getName());
        props.setProperty(Environment.CURRENT_SESSION_CONTEXT_CLASS, "thread");
        props.setProperty(Environment.STATEMENT_BATCH_SIZE, "1");
        return props;
    }

    public static void removeDBFiles(String dbname) {
        for (String dbFile: new File(".").list()) {
            File f = new File(dbFile);
            if (f.getName().startsWith(dbname)) {
                if (!f.delete()) {
                    throw new RuntimeException("Lock still held on DB file" + f.getName());
                }
            }
        }
    }

    /* (non-Javadoc)
	 * @see junit.framework.TestCase#setUp()
	 */
	@Override
	protected void setUp() throws Exception {
        removeDBFiles(DBNAME);
	}

	private Resource createResource(String dbname) {
		return createResource(dbname, null);
	}
    private HbDataStore createHbDataStore(String dbname) {
		Properties props = getHibernateProperties(dbname);
		EPackage[]packages = new EPackage[]{TestmapPackage.eINSTANCE};
		HbDataStore hbds = HbHelper.INSTANCE.createRegisterDataStore(dbname);
        hbds.setEPackages(packages);
        // set the relational database to use and other properties
        hbds.setHibernateProperties(props);
        hbds.setPersistenceProperties(props);
        hbds.initialize();
        return hbds;
	}
    private Resource createResource(String dbname, String queryString) {
		String uriStr = "hibernate://?" + HibernateResource.DS_NAME_PARAM + "=" + dbname;
		if (queryString != null) {
			uriStr += queryString;
		}
		final URI uri = URI.createURI(uriStr);
        return new HibernateResource(uri);
	}
    
    @SuppressWarnings("unchecked")
	public void testSaving() throws IOException {
	    HbDataStore store = createHbDataStore(DBNAME);
	    TableWithMap someResource;
		Resource res = null;
		try {
			res = createResource(DBNAME);
			res.load(Collections.EMPTY_MAP);
			someResource = TestmapFactory.eINSTANCE.createTableWithMap();
	        someResource.getMembers().put("foo", "bar");
			res.getContents().add(someResource);
			res.save(null);
			validateNameValuePair();
		} finally {
			if (res != null) {
				try {
					res.unload();
				} catch (SessionException e) {
					// Ignore...
				}
			}
			store.close();
		}
	}
    
    private void validateNameValuePair() throws IOException {
		Resource res = createResource(DBNAME, "&query1=FROM TableWithMap");
		try {
			res.load(Collections.EMPTY_MAP);
			Assert.assertNotNull(res.getContents());
			Assert.assertEquals(1, res.getContents().size());
	        TableWithMap someResource = (TableWithMap)res.getContents().get(0);
	        Assert.assertEquals(1, someResource.getMembers().size());
	        Assert.assertTrue(someResource.getMembers().containsKey("foo"));
	        Assert.assertTrue(someResource.getMembers().containsValue("bar"));
		} finally {
			if (res != null) {
				res.unload();
			}
		}
    }
}
