package eclipse.service.part;

import java.io.File;
import java.net.URL;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Path;
import org.eclipse.e4.ui.di.Focus;
import org.eclipse.e4.ui.model.application.ui.basic.MPart;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.resource.FontDescriptor;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.jface.viewers.DelegatingStyledCellLabelProvider;
import org.eclipse.jface.viewers.DelegatingStyledCellLabelProvider.IStyledLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.osgi.framework.Bundle;
import org.osgi.framework.FrameworkUtil;

import jakarta.annotation.PostConstruct;
import jakarta.inject.Inject;

public class FileBrowserPart {
    private TreeViewer viewer;
    
    @Inject
    private MPart part;
    
    @PostConstruct
    public void createControls(Composite parent) {
    

        viewer = new TreeViewer(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL);
        viewer.setContentProvider(new ViewContentProvider());
        viewer.setLabelProvider(new DelegatingStyledCellLabelProvider(
                new ViewLabelProvider(createImageDescriptor("icons/folder.png"))));
        viewer.setInput(File.listRoots());
//        File initialPath = new File("D:/gitcode/moging/Mogine");
//        viewer.setInput(initialPath.listFiles());

        // Set layout and layout data to adjust the tree viewer's spacing
        GridLayoutFactory.fillDefaults().applyTo(parent);
        GridDataFactory.fillDefaults().grab(true, true).applyTo(viewer.getTree());

        // Set resourceManager to manage the resource ： font
        ResourceManager resourceManager = new LocalResourceManager(JFaceResources.getResources(), parent);
        viewer.getTree().setFont(resourceManager.createFont(FontDescriptor.createFrom("SimHei", 12,SWT.NORMAL)));

        // Adjust item height
        adjustItemHeight(viewer);
    }
    
    
    private void adjustItemHeight(TreeViewer viewer) {
        viewer.getTree().addListener(SWT.MeasureItem, event -> {
            event.height = 35;  // Adjust the height as needed
        });
    }

    private ImageDescriptor createImageDescriptor(String imagePath) {
        Bundle bundle = FrameworkUtil.getBundle(ViewLabelProvider.class);
        URL url = FileLocator.find(bundle, new Path(imagePath), null);
        return ImageDescriptor.createFromURL(url);
    }

    class ViewContentProvider implements ITreeContentProvider {
        public void inputChanged(Viewer v, Object oldInput, Object newInput) {
        }

        @Override
        public void dispose() {
        }

        @Override
        public Object[] getElements(Object inputElement) {
            return (File[]) inputElement;
        }

        @Override
        public Object[] getChildren(Object parentElement) {
            File file = (File) parentElement;
            return file.listFiles();
        }

        @Override
        public Object getParent(Object element) {
            File file = (File) element;
            return file.getParentFile();
        }

        @Override
        public boolean hasChildren(Object element) {
            File file = (File) element;
            if (file.isDirectory()) {
                return true;
            }
            return false;
        }

    }

    class ViewLabelProvider extends LabelProvider implements IStyledLabelProvider {
        
        private ImageDescriptor directoryImage;
        private ResourceManager resourceManager;

        public ViewLabelProvider(ImageDescriptor directoryImage) {
            this.directoryImage = directoryImage;
        }

        @Override
        public StyledString getStyledText(Object element) {
            if(element instanceof File) {
                File file = (File) element;
                StyledString styledString = new StyledString(getFileName(file));
                String[] files = file.list();
//                if (files != null) {
//                    styledString.append(" ( " + files.length + " ) ",
//                            StyledString.COUNTER_STYLER);
//                }
                return styledString;
            }
            return null;
        }
        
        @Override
        public Image getImage(Object element) {
            if(element instanceof File) {
                if(((File) element).isDirectory()) {
                    return getResourceManager().createImage(directoryImage);
                }
            }
            
            return super.getImage(element);
        }
        
        @Override
        public void dispose() {
            // garbage collect system resources
            if(resourceManager != null) {
                resourceManager.dispose();
                resourceManager = null;
            }
        }
        
        protected ResourceManager getResourceManager() {
            if(resourceManager == null) {
                resourceManager = new LocalResourceManager(JFaceResources.getResources());
            }
            return resourceManager;
        }

        private String getFileName(File file) {
            String name = file.getName();
            return name.isEmpty() ? file.getPath() : name;
        }
    }
    
    @Focus
    public void setFocus() {
        viewer.getControl().setFocus();
    }
}
