package org.eclipse.epsilon.examples.standalone.evl;

import java.io.File;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceFactoryImpl;
import org.eclipse.epsilon.common.module.ModuleElement;
import org.eclipse.epsilon.common.parse.AST;
import org.eclipse.epsilon.common.parse.problem.ParseProblem;
import org.eclipse.epsilon.emc.emf.InMemoryEmfModel;
import org.eclipse.epsilon.eol.exceptions.models.EolModelElementTypeNotFoundException;
import org.eclipse.epsilon.eol.exceptions.models.EolModelNotFoundException;
import org.eclipse.epsilon.evl.EvlModule;
import org.eclipse.epsilon.evl.dom.ConstraintContext;
import org.eclipse.epsilon.evl.execute.context.IEvlContext;

/**
 * Demonstrates subclassing EvlModule to only check constraints against a
 * selection of model elements
 */

public class SelectiveEvlModule extends EvlModule {

	protected List<EObject> rootEObjects;

	@Override
	public ModuleElement adapt(AST cst, ModuleElement parentAst) {
		ModuleElement moduleElement = super.adapt(cst, parentAst);
		if (moduleElement instanceof ConstraintContext) {
			return new SelectiveConstraintContext();
		}
		return moduleElement;
	}

	class SelectiveConstraintContext extends ConstraintContext {
		@Override
		public Collection<?> getAllOfSourceKind(IEvlContext context)
				throws EolModelElementTypeNotFoundException, EolModelNotFoundException {
			// Only return a subset containing elements which are
			// descendants of one of the root objects
			return super.getAllOfSourceKind(context).stream().filter(e -> shouldCheck(e)).collect(Collectors.toList());
		}
	}

	protected boolean shouldCheck(Object object) {
		// Only check elements which are decendants of one of the
		// root objects
		return EcoreUtil.isAncestor(rootEObjects, (EObject) object);
	}

	public void setRootEObjects(List<EObject> rootEObjects) {
		this.rootEObjects = rootEObjects;
	}

	public List<EObject> getRootEObjects() {
		return rootEObjects;
	}

	public static void main(String[] args) throws Exception {

		SelectiveEvlModule module = new SelectiveEvlModule();

		ResourceSetImpl resourceSet = new ResourceSetImpl();
		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap().put("*", new XMIResourceFactoryImpl());
		Resource resource = resourceSet.createResource(URI.createURI("foo.ecore"));
		resourceSet.getPackageRegistry().put(EcorePackage.eNS_URI, EcorePackage.eINSTANCE);

		// Create 2 classes C1 and C2 with one attribute each
		EClass c1 = EcoreFactory.eINSTANCE.createEClass();
		c1.setName("C1");
		EAttribute a1 = EcoreFactory.eINSTANCE.createEAttribute();
		a1.setName("a1");
		c1.getEStructuralFeatures().add(a1);

		EClass c2 = EcoreFactory.eINSTANCE.createEClass();
		c2.setName("C2");
		EAttribute a2 = EcoreFactory.eINSTANCE.createEAttribute();
		a2.setName("a2");
		c2.getEStructuralFeatures().add(a2);

		resource.getContents().add(c1);
		resource.getContents().add(c2);

		InMemoryEmfModel model = new InMemoryEmfModel(resource);

		// Constraint for EAttibute
		module.parse(new File("mainRules.evl"));
		if (module.getParseProblems().size() > 0) {
			for (ParseProblem problem : module.getParseProblems()) {
				System.out.println(problem.toString());
			}
			return;
		}
		
		
		module.getContext().getModelRepository().addModel(model);

		// Only elements under C1 (i.e. attribute a1) should be checked
		module.setRootEObjects(Arrays.asList(c1));

		module.execute();
	}

}