/*
 *****************************************************************************
 * Copyright (c) 2004, 2005 Actuate Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Actuate Corporation  - initial API and implementation
 *
 ******************************************************************************
 */

package org.eclipse.birt.data.engine.odaconsumer;

import java.sql.Types;
import java.util.logging.Level;

import org.eclipse.birt.data.engine.core.DataException;
import org.eclipse.birt.data.engine.i18n.ResourceConstants;
import org.eclipse.birt.data.oda.IResultSetMetaData;
import org.eclipse.birt.data.oda.OdaException;

/**
 * ResultSetMetaData contains the result set metadata retrieved during 
 * runtime.
 */
public class ResultSetMetaData
{
	private IResultSetMetaData m_metadata;
	private String m_driverName;
	private String m_dataSetType;

	// trace logging variables
	private static String sm_className = ResultSetMetaData.class.getName();
	private static String sm_loggerName = ConnectionManager.sm_packageName;
	private static LogHelper sm_logger = LogHelper.getInstance( sm_loggerName );

	ResultSetMetaData( IResultSetMetaData metadata, String driverName,
					   String dataSetType )
	{
		String methodName = "ResultSetMetaData";
		if( sm_logger.isLoggingEnterExitLevel() )
		    sm_logger.entering( sm_className, methodName, 
            		new Object[] { metadata, driverName, dataSetType } );

		m_metadata = metadata;
		m_driverName = driverName;
		m_dataSetType = dataSetType;

	    sm_logger.exiting( sm_className, methodName, this );
	}
	
	/**
	 * Returns the number of columns in the corresponding result set.
	 * @return	the number of columns in the result set.
	 * @throws DataException	if data source error occurs.
	 */
	public int getColumnCount( ) throws DataException
	{
	    String methodName = "getColumnCount";
		try
		{
			return m_metadata.getColumnCount( );
		}
		catch( OdaException ex )
		{
		    sm_logger.logp( Level.SEVERE, sm_className, methodName,
            				"Cannot get column count.", ex );
			throw new DataException( ResourceConstants.CANNOT_GET_COLUMN_COUNT, ex );
		}
		catch( UnsupportedOperationException ex )
		{
		    sm_logger.logp( Level.SEVERE, sm_className, methodName,
            				"Cannot get column count.", ex );
			throw new DataException( ResourceConstants.CANNOT_GET_COLUMN_COUNT, ex );
		}
	}
	
	/**
	 * Returns the column name at the specified column index.
	 * @param index	the column index.
	 * @return		the column name at the specified column index.
	 * @throws DataException	if data source error occurs.
	 */
	public String getColumnName( int index ) throws DataException
	{
	    String methodName = "getColumnName";
		try
		{
			return m_metadata.getColumnName( index );
		}
		catch( OdaException ex )
		{
		    sm_logger.logp( Level.SEVERE, sm_className, methodName,
            				"Cannot get column name.", ex );
			throw new DataException( ResourceConstants.CANNOT_GET_COLUMN_NAME, ex, 
			                         new Object[] { new Integer( index ) } );
		}
		catch( UnsupportedOperationException ex )
		{
		    sm_logger.logp( Level.WARNING, sm_className, methodName,
    						"Cannot get column name.", ex );
		    return "";
		}
	}
	
	/**
	 * Returns the column label at the specified column index.
	 * @param index	the column index.
	 * @return		the column label at the specified column index.
	 * @throws DataException	if data source error occurs.
	 */
	public String getColumnLabel( int index ) throws DataException
	{
	    String methodName = "getColumnLabel";
		try
		{
			return m_metadata.getColumnLabel( index );
		}
		catch( OdaException ex )
		{
		    sm_logger.logp( Level.SEVERE, sm_className, methodName,
    						"Cannot get column label.", ex );
			throw new DataException( ResourceConstants.CANNOT_GET_COLUMN_LABEL, ex, 
			                         new Object[] { new Integer( index ) } );
		}
		catch( UnsupportedOperationException ex )
		{
		    sm_logger.logp( Level.INFO, sm_className, methodName,
    						"Cannot get column label.", ex );
		    return "";
		}
	}
	
	/**
	 * Returns the <code>java.sql.Types</code> type at the specified column index.
	 * @param index	the column index.
	 * @return		the <code>java.sql.Types</code> type at the specified column index.
	 * @throws DataException	if data source error occurs.
	 */
	public int getColumnType( int index ) throws DataException
	{
	    String methodName = "getColumnType";
	    
		int nativeType = doGetColumnType( index );
		int odaType = 
			DriverManager.getInstance().getNativeToOdaMapping( m_driverName, 
															   m_dataSetType,
															   nativeType );
		if( odaType != Types.INTEGER &&
			odaType != Types.DOUBLE &&
			odaType != Types.CHAR &&
			odaType != Types.DECIMAL &&
			odaType != Types.DATE &&
			odaType != Types.TIME &&
			odaType != Types.TIMESTAMP &&
			odaType != Types.NULL &&
			odaType != Types.BINARY)
			assert false;	// exception is now thrown by DriverManager
		
		if( sm_logger.isLoggable( Level.FINEST ) )
		    sm_logger.logp( Level.FINEST, sm_className, methodName, 
		            		"Column at index {0} has ODA data type {1}.",
		            		new Object[] { new Integer( index ), new Integer( odaType ) } );

		return odaType;
	}
	
	/**
	 * Returns the native type name at the specified column index.
	 * @param index	the column index.
	 * @return		the native type name.
	 * @throws DataException	if data source error occurs.
	 */
	public String getColumnNativeTypeName( int index ) throws DataException
	{
	    String methodName = "getColumnNativeTypeName";
		try
		{
			return m_metadata.getColumnTypeName( index );
		}
		catch( OdaException ex )
		{
		    sm_logger.logp( Level.SEVERE, sm_className, methodName,
							"Cannot get column native type name.", ex );
			throw new DataException( ResourceConstants.CANNOT_GET_COLUMN_NATIVE_TYPE_NAME, ex, 
									 new Object[] { new Integer( index ) } );
		}
		catch( UnsupportedOperationException ex )
		{
		    sm_logger.logp( Level.WARNING, sm_className, methodName,
							"Cannot get column native type name.", ex );
		    return "";
		}
	}

	private int doGetColumnType( int index ) throws DataException
	{
	    String methodName = "doGetColumnType";
		try
		{
			return m_metadata.getColumnType( index );
		}
		catch( OdaException ex )
		{
		    sm_logger.logp( Level.SEVERE, sm_className, methodName,
							"Cannot get column native type code.", ex );
			throw new DataException( ResourceConstants.CANNOT_GET_COLUMN_TYPE, ex, 
			                         new Object[] { new Integer( index ) } );
		}
		catch( UnsupportedOperationException ex )
		{
		    sm_logger.logp( Level.SEVERE, sm_className, methodName,
							"Cannot get column native type code.", ex );
			throw new DataException( ResourceConstants.CANNOT_GET_COLUMN_TYPE, ex, 
			                         new Object[] { new Integer( index ) } );			
		}
	}
	
	Class getColumnTypeAsJavaClass( int index ) throws DataException
	{
		int odaType = getColumnType( index );
		return DataTypeUtil.toTypeClass( odaType );
	}
}
