/*
 * Created on 16.9.2006. - last revision: 2.4.2007.
 * (C) 2007. Milan Milanovic, GOOD OLD AI Laboratory, Faculty of Organizational Sciences, University of Belgrade.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  (See the file "COPYING" that is included with this source distribution.)
 */

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.atl.eclipse.engine.AtlModelHandler;
import org.atl.engine.extractors.Extractor;
import org.atl.engine.extractors.ebnf.EBNFExtractor;
import org.atl.engine.extractors.xml.XMLExtractor;
//import org.atl.engine.injectors.ebnf.EBNFInjector2;
import org.atl.engine.injectors.xml.XMLInjector;
import org.atl.engine.repositories.mdr4atl.ASMMDRModel;
import org.atl.engine.repositories.mdr4atl.AtlMDRModelHandler;
import org.atl.engine.vm.nativelib.ASMModel;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Main class for running ATL transformations programatically (template)
 * TODO: Add support for running AM3 Ant tasks
 * @author Milan Milanovic (milan [at] milanovic.org)
 * @version 2.3
 */
public class ATLTransformations {

	// Stop on error in EBNF Injection
	//private final static boolean stopOnError = !false;

	// Relative location of transformations
	String transformationsLocation = "transformations/";

	// Relative location of metamodels
	String transformationsMetamodels = "metamodel/";

	// URL's to compiled ATL transformations (an example of OneMM to OtherMM transformation)
	private URL OneMM2OtherMMurl = ATLTransformations.class.getResource(transformationsLocation + "Transformation.asm");
	// Add here URL's to your other compiled transformations...

	// TCS specific (MM is pseudoname for some your metamodel which is used for TCS injection/extraction)
	//private URL MM_tcs = ATLTransformations.class.getResource(transformationsMetamodels + "MM/MM-TCS.xmi");
	//private URL TCS = ATLTransformations.class.getResource(transformationsMetamodels + "TCS/TCS.xmi"); // uses MDR (change to .ecore for EMF)

	// MDR model handler
	private AtlModelHandler mdramh = null;

	// EMF model handler
	//private AtlModelHandler emfamh = null;

	// MDR meta-metamodels
	private ASMModel oneMM_MDR = null;
	private ASMModel otherMM_MDR = null;
	private ASMModel xmlMDRmm = null; // XML metamodel is needed because of XML injection/extraction
	// Add here objects for you metamodels (MDR - MOF-1.4)...

	// For TCS (MDR) support
	private ASMModel tcsMDRmm = null;
	private ASMModel mmTcsMDRmm = null;

	// Used for keeping metamodel names
	Map MDRMetaModels = new HashMap();

	// EMF meta-metamodels
	//private ASMModel oneMM_EMF = null;
	//private ASMModel otherMM_EMF = null;
	//private ASMModel xmlEMFmm = null;
	// Add here objects for you metamodels (EMF - Ecore)

	// For TCS (EMF) support
	//private ASMModel tcsEMFmm = null;
	//private ASMModel mmTcsEMFmm = null;

	// General meta-models
	//private ASMModel pbmm = null;

	// Instance of ATLTransformations class (Singleton)
    private static ATLTransformations atlTransformations = null;

	// Markers for Problem metamodel
	//private MarkerMaker markerMaker;

	// Constructor
	public ATLTransformations() {
		initMDR(); // Initialize MDR based metamodels
	    //initEMF(); // Initialize EMF based metamodels - if needed
	} // -- end of constructor

	/////////////////////////////////////////////////////////////////////////////////////////////
	//// General ATL transformation initialization with domain metamodels specific initialization
	/////////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * Initialize MDR model handler and MOF based metamodels
     */
	private void initMDR() {

		if(mdramh == null) { // if MDR is not initialized

			// Initialize MDR model handler
			mdramh = getDefaultHandler("MDR");

			// URL's to MOF-1.4 (XMI) metamodels
			URL onemmurl = ATLTransformations.class.getResource(transformationsMetamodels + "UML1.4.xmi");
			URL xmlmmurl = ATLTransformations.class.getResource(transformationsMetamodels + "xml/xml.xmi");
			URL othermmurl = ATLTransformations.class.getResource(transformationsMetamodels + "UML1.4.xmi");
			// Add here URL's to your metamodels (MDR - MOF-1.4)...

			try {
				// Get metamodels
				oneMM_MDR = mdramh.loadModel("OneMM", mdramh.getMof(), onemmurl.openStream()); 			// ONE MM
				xmlMDRmm = mdramh.loadModel("XML", mdramh.getMof(), xmlmmurl.openStream()); 			// XML MM
				otherMM_MDR = mdramh.loadModel("OtherMM", mdramh.getMof(), othermmurl.openStream()); 	// Other MM
				// Load your metamodels here...

				// Only for TCS output!!
				//tcsMDRmm = mdramh.loadModel("TCS", mdramh.getMof(), TCS.openStream()); 			// TCS MM
				//mmTcsMDRmm = mdramh.loadModel("MM.tcs", tcsMDRmm, MM_tcs.openStream()); 		// MM-TCS MM

				// Put metamodels in map for easy lookup
				MDRMetaModels.put("OneMM", oneMM_MDR);
				MDRMetaModels.put("XML", xmlMDRmm);
				MDRMetaModels.put("OtherMM", otherMM_MDR);
				//MDRMetaModels.put("TCS", tcsMDRmm);
				//MDRMetaModels.put("MM-TCS", mmTcsMDRmm);
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	} // -- end of initMDR

	/**
	 * Initialize EMF model handler and Ecore based metamodels
     */
/*	private void initEMF() {

		if(emfamh == null) { // if EMF is not initialized

			// Initialize EMF model handler
			emfamh = AtlModelHandler.getDefault(AtlModelHandler.AMH_EMF);

			// URL's to Ecore (XMI) metamodels
			URL onemmurl = ATLTransformations.class.getResource(transformationsMetamodels + "oneMM/oneMM.ecore");
			URL xmlmmurl = ATLTransformations.class.getResource(transformationsMetamodels + "xml/xml.ecore");
			URL othermmurl = ATLTransformations.class.getResource(transformationsMetamodels + "otherMM/otherMM.ecore");
			// Add here URL's to your metamodels (EMF - Ecore)...

			try {
				// Get metamodels
				oneMM_EMF = emfamh.loadModel("OneMM", emfamh.getMof(), onemmurl.openStream());
				xmlEMFmm = emfamh.loadModel("XML", emfamh.getMof(), xmlmmurl.openStream());
				otherMM_EMF = emfamh.loadModel("OtherMM", emfamh.getMof(), othermmurl.openStream());
				// Load your metamodels here...

				// Only for TCS output!!
				//tcsEMFmm = emfamh.loadModel("TCS", emfamh.getMof(), TCS.openStream()); 		// TCS MM
				//mmTcsEMFmm = emfamh.loadModel("MM.tcs", tcsEMFmm, MM_tcs.openStream()); 		// MM-TCS MM

			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		pbmm = emfamh.getBuiltInMetaModel("Problem");
		markerMaker = new MarkerMaker();
	} // -- end of initEMF
*/
   /**
    * Returns default model handler (currently supported MDR and EMF)
    * @param repository String which represents model handler
    * @return default repository for input (EMF or MDR)
    */
	private AtlModelHandler getDefaultHandler(String repository) {
	    AtlModelHandler amh = null;
	    try {
	         amh = AtlModelHandler.getDefault(repository);
	    } catch (Throwable e) {
	         if ("MDR".equals(repository)) {
	             AtlModelHandler.registerDefaultHandler("MDR", new AtlMDRModelHandler()); // register
	             amh = AtlModelHandler.getDefault(repository);
	         } else {
	             throw new RuntimeException(e);
	         }
	     }
	     return amh;
    } // -- end of getDefaultHandler

   /**
	 * Returns static instance of ATLTransformations class (Singleton)
	 * @return default repository for input (EMF or MDR)
	 */
    public static ATLTransformations getATLTransformations() {
		if(atlTransformations  == null)
			atlTransformations = new ATLTransformations();

		return atlTransformations;
	}

    /**
     * Check if input XML file is well-formed - helper method
     * @param file path to input XML file or XML code
     * @param isFile if it is true then it is path to file, otherwise it is String that contains XML code
     * @return true if input XML file is well formed, otherwise is false
     */
    private boolean checkForWellFormedness(String file, boolean isFile)
	{
		SAXParser saxParser = null;
		DefaultHandler dh = null;
		InputStream in = null;

		// init parser
		try {
			SAXParserFactory spfactory = SAXParserFactory.newInstance();
			saxParser = spfactory.newSAXParser();
			dh = new DefaultHandler();
		}
		catch(Exception e) {
			System.out.println("Cannot initialize SAX parser.");
			e.printStackTrace();
			return false;
		}

		// parse the XML document using SAX parser
		try {
			if (isFile == true) { // it is file
				in = new FileInputStream(file);
			}
			else { // it is String that contains XML code
				byte currentXMLBytes[] = file.getBytes(); // Get bytes from input String
				in = new ByteArrayInputStream(currentXMLBytes);
			}
			saxParser.parse(in, dh); // SAXException, IOException
		}
		catch(SAXParseException spe) { // Document is not well-formed: Error
			System.out.println("Document is not well-formed.");
			return false;
		}
		catch(SAXException se) { // (*)
			System.out.println("Error while parsing input XML file: " + file);
			se.printStackTrace();
			return false;
		}
		catch(FileNotFoundException f) {
			System.out.println("Error: File is not found");
			return false;
		}
		catch(IOException ioe) {
			System.out.println("Cannot read file.");
			return false;
		}
		return true;
	} // -- end of checkForWellFormedness

    /**
     * Inject input XML file to XML model (instance of XML metamodel - MOF 1.4)
     * Note: if you want to use EMF, just change 'mdramh' to 'emfamh' and 'xmlMDRmm' to 'xmlEMFmm'
     *       in this method, and also in other methods if needed.
     * @param file name of file from which XML model will be created
     * @return created XML model from input file
     */
	public ASMModel injectXMLModelFromFile(String file) {
		initMDR(); // if MDR is not initialized, do it now !

		ASMModel ret = mdramh.newModel("IN", xmlMDRmm); // create new output model with XML metamodel (MOF-1.4)

		XMLInjector xmli = new XMLInjector(); // create XML injector

		Map parameters = Collections.EMPTY_MAP; // no parameters

		InputStream in = null;

		try {
			in = new FileInputStream(file);

			xmli.inject(ret, in, parameters); // Do injection !

		} catch(FileNotFoundException f) {
			System.out.println("Error: File is not found");
		} catch(IOException io) {
			System.out.println("Error: in injection of XML file");
		}

		return ret;
	} // -- end of injectXMLModelFromFile

    /**
     * Convert input String which contains XML code to XML model (instance of XML metamodel - MOF 1.4)
     * @param inputString String which contains XML
     * @return created XML model from in XML code
     */
	public ASMModel injectXMLModelFromString(String inputString) {
		initMDR(); // if MDR is not initialized, do it now !

		// Get bytes from input String
		byte currentXMLBytes[] = inputString.getBytes();

		InputStream in = new ByteArrayInputStream(currentXMLBytes);

		ASMModel ret = mdramh.newModel("IN", xmlMDRmm); // create new output model with XML metamodel (MOF-1.4)

		XMLInjector xmli = new XMLInjector(); // create XML injector

		Map parameters = Collections.EMPTY_MAP; // no parameters

		try {
			xmli.inject(ret, in, parameters); // Do injection !
		} catch(IOException io){
			io.printStackTrace();
		}

		return ret;
	} // -- end of injectXMLModelFromFile

    /**
     * Extract input XML model (instance of XML metamodel - MOF 1.4) to XML file
     * @param model input XML model which will be extracted to XML file
     * @param file output XML file in which input XML model will be extracted
     */
	public void extractXMLModelToFile(ASMModel model, String file) {
		initMDR();

		OutputStream out = null;

		Map parameters = Collections.EMPTY_MAP;

		XMLExtractor xmle = new XMLExtractor();

		try {
			out = new FileOutputStream(file);

			xmle.extract(model, out, parameters); // Do extraction !

			out.flush(); out.close(); // close stream
		} catch(FileNotFoundException f) {
			System.out.println("Error: File is not found");
		} catch(IOException io) {
			System.out.println("Error: in extraction of XML model to XML file");
		}
	} // -- end of extractXMLModelToFile

    /**
     * Extract input XML model (instance of XML metamodel - MOF 1.4) to String
     * @param model input XML model which will be extracted to XML file
     * @return input model returned as String
     */
	public String extractXMLModelToString(ASMModel model) {
		initMDR();

		OutputStream out = null; String ret = null;

		Map parameters = Collections.EMPTY_MAP;

		XMLExtractor xmle = new XMLExtractor();

		try {
			out = new ByteArrayOutputStream();

			xmle.extract(model, out, parameters); // Do extraction !

			ret = out.toString();

			out.flush(); out.close(); // close stream
		} catch(FileNotFoundException f) {
			System.out.println("Error: File is not found");
		} catch(IOException io) {
			System.out.println("Error: in extraction of XML model to XML file");
		}
		return ret;
	} // -- end of extractXMLModelToFile

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// An example EBNF inject/extract methods for some MM metamodel (this should be changed for every metamodel)
	// Note: you should have generated MM-ebnfinjector.jar and putted
	//       in into classpath with other libs
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
     * Extract input MM model (instance of some MM metamodel - MOF 1.4) to File
	 * @param MMModel input MM model which will be extracted to File
	 * @param file path to the file in which will be extracted input MM model
	 */
    public void saveMMModelToFile(ASMModel MMModel, String file) {
		initMDR();

		OutputStream out = null;

		Extractor ext = new EBNFExtractor();
		Map params = new HashMap();
		params.put("format", MDRMetaModels.get("MM-TCS"));
		params.put("indentString", "\t");

		try {
			out = new FileOutputStream(file);
			ext.extract(MMModel, out, params); // do it!

		} catch(Exception e) {
			e.printStackTrace();
		}
	} // -- end of saveMMModelToFile

	/**
	 * Deserializes a MM model from a File. (works with MDR, but if it is needed, just change to EMF)
	 * @param file is the MM file.
	 * @return the MM model that is injected from input MM code.
	 */
	 public ASMModel getMMFromFile(String file) {
		initMDR(); // if MDR (and EMF) is not initialized, do it now !

		ASMModel ret = mdramh.newModel("IN", oneMM_MDR); // create new output model with OneMM (or some your MM) metamodel

		//EBNFInjector2 ebnfi = new EBNFInjector2();
		InputStream in = null;

		try {
			in = new FileInputStream(file);

			//Class lexer = MMLexer.class; // This are your generated classes
			//Class parser = MMParser.class;

			//ebnfi.performImportation(oneMM_MDR, ret, in, "MM", lexer, parser);

			in.close();

			} catch(FileNotFoundException f) {
				System.out.println("Error: File is not found");
			} catch(IOException io) {
				System.out.println("Error: in injection of MM file");
			} catch(Exception e) {
				e.printStackTrace();
			}

			// TODO: Check for problems with: int nbPbs = markerMaker.applyMarkers(file, pbs); ...

			return ret;
	} // -- end of getMMFromFile

   /**
     * Launch ATL transformation
     * @param modelHandler model handler which is used for creating output model
     * @param transformation URL to compiled ATL transformation
     * @param inputModel input model for transformation
     * @param inputMetamodel input metamodel for transformation
     * @param outputMetamodel output metamodel for transformation
     * @param inParams additional parameters for transformation
     * @param inLibs additional libraries for transformation
     */
	public ASMModel runATLTransformation(AtlModelHandler modelHandler, URL transformation, ASMModel inputModel,
									     ASMModel inputMetamodel, ASMModel outputMetamodel, Map inParams, Map inLibs) {
		initMDR(); // Initialize MDR model handler

		ASMModel ret = null; // return model

		// Set launch configuration
		Map models = new HashMap();
		models.put(inputMetamodel.getName(), inputMetamodel); // input metamodel
		models.put(outputMetamodel.getName(), outputMetamodel); // output metamodel
		models.put("IN", inputModel); // input model
		ret = modelHandler.newModel("OUT", outputMetamodel); // output model
		models.put("OUT", ret);

		Map params = inParams; // Parameters
		Map libs = inLibs; 	   // Libraries

		// Launch ATL transformation
		AtlLauncher.getDefault().launch(transformation, libs, models, params);

		return ret;
	} // -- end of runATLTransformation

    /**
     * Load model from file (XMI) as MOF-1.4
     * @param file name of file which containts input model
     * @param inputMetamodel input metamodel (as String)
     * @return model which is constructed from input file
     */
	public ASMModel loadMDRModelFromFile(String file, String inputMetamodel) {
		// Initialize MDR
		initMDR();

		// Get metamodel from map
		ASMModel metamodel = (ASMModel)MDRMetaModels.get(inputMetamodel);

		InputStream in = null;

		try {
			// Create input stream from file
			in = new FileInputStream(file);

		} catch(FileNotFoundException e) {
			System.out.println("Error: Input file is not found!");
			return null;
		}

		return mdramh.loadModel(file, metamodel, in);
	} // -- end of loadMDRModelFromFile

   /**
     * Save model to file (XMI) as MOF-1.4
     * @param inputModel input model for saving to file
     * @param file name of file in which model will be saved
     */
	public void saveMDRModelToFile(ASMModel inputModel, String file) {
		// Bridge actual problem of saving model outside of Eclipse environment
		ASMMDRModel model = (ASMMDRModel)inputModel;

		OutputStream out = null;
		try {
			// Create output stream
			out = new FileOutputStream(file);

			// Save model
			model.save(out);

			out.flush(); out.close();

		} catch(FileNotFoundException f) {
			System.out.println("Error: output file can't be created");
		} catch(IOException io) {
			System.out.println("Error: model can't be saved to file");
		}
	} // -- end of saveMDRModelToFile

   /**
     * Converts MDR ASMModel to String
     * @param MOFmodel input model for converting to string
     * @return String which containts input model
     */
	public String MOFModelToString(ASMModel MOFmodel) {
		// Create output stream
		ByteArrayOutputStream out = new ByteArrayOutputStream();

		String ret = null;

		try {
			// Save model
			((ASMMDRModel)MOFmodel).save(out);

			out.flush();

			ret = out.toString();

			out.close();

		} catch(IOException io) {
			System.out.println("Error: model to string");
		}
		return ret;
	} // -- end of MOFModelToString

	//////////////////////////////////////////////////////////////////////////////////
	//// Specific transformations
	//////////////////////////////////////////////////////////////////////////////////

    /**
     * Run OneMM to OtherMM (example) ATL transformation (OneMM2OtherMM).
     * Note: MDR is used as model handler, if you want to use EMF, just change 'mdramh' to 'emfamh'
     * @param oneModel input ONE model
     * @return output (cleaned) OtherMM model
     */
	public ASMModel getOtherMMFromOneMM(ASMModel oneModel) {

		// If you need library, you can add it in following way:
		// e.g., Map libs = new HashMap();
		//  	 libs.put("XMLHelpers", XMLHelpersurl);
		// And you just have to add this 'libs' object instead of the last Collections.EMPTY_MAP below

		// Run transformation and return output model
		return runATLTransformation(mdramh, OneMM2OtherMMurl, oneModel, oneMM_MDR, otherMM_MDR, Collections.EMPTY_MAP, Collections.EMPTY_MAP);
	} // -- end of getCleanedXMLFromXML

   /**
	 * Transform input One (XML-based) file to output Other (XML-based) file - an example with injection/extraction
	 * @param InputOneFile name of input One XML file
	 * @param OutputOtherFile name of output Other XML file
	 * @return Result of running this transformation
	 */
	public String transformOnetoOther(String InputOneFile, String OutputOtherFile) {
		// Check is input String well-formed (XML)
		if (!checkForWellFormedness(InputOneFile, true)) return new String("Document is not well-formed.");

		ASMModel xmlModel = injectXMLModelFromFile(InputOneFile); 	// One XML -> XML model
		ASMModel otherModel = getOtherMMFromOneMM(xmlModel); 		// One XML -> Other
		extractXMLModelToFile(otherModel, OutputOtherFile); 		// Other -> Other XML
		return new String("OK");
	} // -- end of transformOnetoOther

	//////////////////////////////////////////////////////////////////////////////////
	//// Main method - used only for testing this class
	//////////////////////////////////////////////////////////////////////////////////
	public static void main(String [] arguments) {

		// Create new instance of this class
		ATLTransformations transformation = ATLTransformations.getATLTransformations();

		// Transform input file to output file
		String message = transformation.transformOnetoOther("models/Input.xml", "models/output.xml"); // Message is OK if all goes well
																				     // or document is not well-formed
		System.out.println(message);
	}

} // -- end of ATLTransformations class