/*******************************************************************************
 * Copyright (c) 2008 The University of York.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Dimitrios Kolovos - initial API and implementation
 ******************************************************************************/
package org.eclipse.epsilon.examples.standalone;

import java.io.File;
import java.net.URISyntaxException;
import java.util.List;

import org.eclipse.emf.common.util.URI;
import org.eclipse.epsilon.common.parse.problem.ParseProblem;
import org.eclipse.epsilon.common.util.StringProperties;
import org.eclipse.epsilon.emc.emf.EmfModel;
import org.eclipse.epsilon.eol.IEolExecutableModule;
import org.eclipse.epsilon.eol.exceptions.EolRuntimeException;
import org.eclipse.epsilon.eol.exceptions.models.EolModelLoadingException;
import org.eclipse.epsilon.eol.execute.context.Variable;
import org.eclipse.epsilon.eol.models.IModel;
import org.eclipse.epsilon.etl.EtlModule;

public abstract class EpsilonStandaloneExample {

	protected IEolExecutableModule module;

	protected Object result;

	public abstract IEolExecutableModule createModule();

	public abstract String getSource() throws Exception;

	public abstract List<IModel> getModels() throws Exception;

	public void postProcess() {
	};

	public void preProcess() {
	};

	public void execute(Variable... parameters) throws Exception {

		module = new EtlModule();
		module.parse(getFile(getSource()));

		if (module.getParseProblems().size() > 0) {
			System.err.println("Parse errors occured...");
			for (ParseProblem problem : module.getParseProblems()) {
				System.err.println(problem.toString());
			}
			System.exit(-1);
		}

		for (IModel model : getModels()) {
			module.getContext().getModelRepository().addModel(model);
		}

		module.getContext().getFrameStack().put(parameters);

		preProcess();
		result = execute(module);
		postProcess();

		module.getContext().getModelRepository().dispose();
	}


	protected Object execute(IEolExecutableModule module)
			throws EolRuntimeException {
		return module.execute();
	}

	protected EmfModel createEmfModel(String name, String model,
			String metamodel, boolean readOnLoad, boolean storeOnDisposal)
			throws EolModelLoadingException, URISyntaxException {
		
		EmfModel emfModel = new EmfModel();
		StringProperties properties = new StringProperties();
		properties.put(EmfModel.PROPERTY_NAME, name);
		properties.put(EmfModel.PROPERTY_FILE_BASED_METAMODEL_URI,
				getFile(metamodel).toURI().toString());
		properties.put(EmfModel.PROPERTY_MODEL_URI, getFile(model).toURI()
				.toString());
		properties.put(EmfModel.PROPERTY_READONLOAD, readOnLoad + "");
		properties.put(EmfModel.PROPERTY_STOREONDISPOSAL, storeOnDisposal + "");

		emfModel.load(properties, null);
		return emfModel;
	}

	protected EmfModel createEmfModelByURI(String name, String model,
			String metamodel, boolean readOnLoad, boolean storeOnDisposal)
			throws EolModelLoadingException, URISyntaxException {
		
		EmfModel emfModel = new EmfModel();
		StringProperties properties = new StringProperties();
		properties.put(EmfModel.PROPERTY_NAME, name);
		properties.put(EmfModel.PROPERTY_METAMODEL_URI, metamodel);
		properties.put(EmfModel.PROPERTY_MODEL_URI, getFile(model).toURI()
				.toString());
		properties.put(EmfModel.PROPERTY_READONLOAD, readOnLoad + "");
		properties.put(EmfModel.PROPERTY_STOREONDISPOSAL, storeOnDisposal + "");
		emfModel.load(properties, null);
		return emfModel;
	}

	
	// Here i have caught the piece of code from eunit example to create an empty emfModel
	
	protected EmfModel loadEmptyModel(String name, String metamodelPath)
			throws EolModelLoadingException {
		
		EmfModel model = new EmfModel();
		model.setName(name);
		model.setMetamodelFileUri(getResourceEmfURI(metamodelPath));
		
		model.setModelFileUri(URI.createFileURI(new File("Empty" + name
				+ ".model").getAbsolutePath()));
		model.setReadOnLoad(false);
		model.setStoredOnDisposal(false);
		model.load();
		
		return model;
	}

	private URI getResourceEmfURI(String resourcePath) {
		return URI.createURI(getClass().getResource("/" + resourcePath)
				.toExternalForm());
	}

	// the source model and the source and target metamodels are in the /src folder
	// i do not know why if if i'm looking for these files in the source folder (standalone/models) i can not 
	// find they.
	protected File getFile(String fileName) throws URISyntaxException {
		java.net.URL url = getClass().getResource("/" + fileName);
		java.net.URI binUri = url.toURI();
		java.net.URI uri = null;

		if (binUri.toString().indexOf("bin") > -1) {
			uri = new java.net.URI(binUri.toString().replaceAll("bin", "src"));
		} else {
			uri = binUri;
		}

		return new File(uri);
	}

}
