/*******************************************************************************
 * Copyright (c) 2012 BestSolution.at and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tom Schindl<tom.schindl@bestsolution.at> - initial API and implementation
 *******************************************************************************/
package org.eclipse.fx.ui.workbench.renderers.fx;

import java.util.ArrayList;
import java.util.List;

import javafx.beans.value.ObservableValue;
import javafx.event.ActionEvent;
import javafx.geometry.Orientation;
import javafx.geometry.Point2D;
import javafx.scene.Node;
import javafx.scene.control.Button;
import javafx.scene.control.ButtonBase;
import javafx.scene.control.Label;
import javafx.scene.control.MenuItem;
import javafx.scene.control.SplitMenuButton;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.FlowPane;
import javafx.scene.layout.HBox;
import javafx.stage.Popup;

import javax.inject.Inject;
import javax.inject.Named;

import org.eclipse.e4.ui.model.application.ui.menu.MMenu;
import org.eclipse.e4.ui.model.application.ui.menu.MMenuElement;
import org.eclipse.e4.ui.model.application.ui.menu.MToolItem;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.fx.ui.workbench.renderers.base.BaseMenuRenderer;
import org.eclipse.fx.ui.workbench.renderers.base.BaseRenderer;
import org.eclipse.fx.ui.workbench.renderers.base.widget.WMenu;
import org.eclipse.fx.ui.workbench.renderers.base.widget.WMenuElement;
import org.eclipse.fx.ui.workbench.renderers.base.widget.WMenuSeparator;
import org.eclipse.fx.ui.workbench.renderers.base.widget.WToolItem;
import org.eclipse.fx.ui.workbench.renderers.fx.widget.ToolItemMenu;
import org.eclipse.fx.ui.workbench.renderers.fx.widget.WWidgetImpl;

/**
 * Default renderer for tool item menus
 */
public class DefToolItemMenuRenderer extends BaseMenuRenderer<ToolItemMenu> {

	@Override
	protected Class<? extends WMenu<ToolItemMenu>> getWidgetClass(MMenu item) {
		MToolItem mToolItem = (MToolItem) ((EObject) item).eContainer();
		if (mToolItem.getTags().contains(WToolItem.TAG_POPUP)) {
			return PopupWMenuImpl.class;
		} else {
			return MenuWMenuImpl.class;
		}
	}

	static class MenuWMenuImpl extends WWidgetImpl<ToolItemMenu, MMenu> implements WMenu<ToolItemMenu> {
		private SplitMenuButton button;
		Runnable showingCallback;
		Runnable hidingCallback;

		@Inject
		public MenuWMenuImpl(@Named(BaseRenderer.CONTEXT_DOM_ELEMENT) MMenu domElement) {
			MToolItem item = (MToolItem) ((EObject) domElement).eContainer();
			@SuppressWarnings("unchecked")
			WToolItem<SplitMenuButton> w = (WToolItem<SplitMenuButton>) item.getWidget();
			this.button = (SplitMenuButton) w.getWidget();
			this.button.showingProperty().addListener(this::handleShowingProperty);
		}

		void handleShowingProperty(ObservableValue<? extends Boolean> observable, Boolean oldValue, Boolean newValue) {
			if (newValue.booleanValue()) {
				if (this.showingCallback != null) {
					this.showingCallback.run();
				}
			} else {
				if (this.hidingCallback != null) {
					this.hidingCallback.run();
				}
			}
		}

		@Override
		public void addStyleClasses(List<String> classnames) {
			getWidget().getStyleClass().addAll(classnames);
		}

		@Override
		public void addStyleClasses(String... classnames) {
			getWidget().getStyleClass().addAll(classnames);
		}
		
		@Override
		public void removeStyleClasses(List<String> classnames) {
			getWidget().getStyleClass().removeAll(classnames);
		}

		@Override
		public void removeStyleClasses(String... classnames) {
			getWidget().getStyleClass().removeAll(classnames);
		}

		@Override
		public void setStyleId(String id) {
			getWidget().setId(id);
		}

		@Override
		protected ToolItemMenu createWidget() {
			return new ToolItemMenu(this.button);
		}

		@Override
		protected void setUserData(WWidgetImpl<ToolItemMenu, MMenu> widget) {
			getWidget().setUserData(widget);
		}

		@Inject
		public void setLabel(@Named(ATTRIBUTE_localizedLabel) String label) {
			getWidget().setText(label);
		}

		@Override
		public void addElement(WMenuElement<MMenuElement> widget) {
			getWidget().getItems().add((MenuItem) widget.getWidget());
		}

		@Override
		public void addElement(int idx, WMenuElement<MMenuElement> widget) {
			getWidget().getItems().add(idx, (MenuItem) widget.getWidget());
		}

		@Override
		public void removeElement(WMenuElement<MMenuElement> widget) {
			getWidget().getItems().remove(widget.getWidget());
		}

		@Override
		public void setShowingCallback(Runnable showingCallback) {
			this.showingCallback = showingCallback;
		}

		@Override
		public void setHidingCallback(Runnable hidingCallback) {
			this.hidingCallback = hidingCallback;
		}
	}
	
	static class PopupWMenuImpl extends WWidgetImpl<Popup, MMenu> implements WMenu<ToolItemMenu> {
		Button button;
		Runnable showingCallback;
		Runnable hidingCallback;
		private Popup popup;
		private Label popupLabel;
		private String popupLabelText;
		private FlowPane rowsPane;
		private FlowPane currentRow;
		
		@Inject
		public PopupWMenuImpl(@Named(BaseRenderer.CONTEXT_DOM_ELEMENT) MMenu domElement) {
			MToolItem item = (MToolItem) ((EObject) domElement).eContainer();
			@SuppressWarnings("unchecked")
			WToolItem<SplitMenuButton> w = (WToolItem<SplitMenuButton>) item.getWidget();
			this.button = (Button) w.getWidget();
		}
		
		@Override
		public void addStyleClasses(List<String> classnames) {
			getWidget().getContent().get(0).getStyleClass().addAll(classnames);
		}

		@Override
		public void addStyleClasses(String... classnames) {
			getWidget().getContent().get(0).getStyleClass().addAll(classnames);
		}
		
		@Override
		public void removeStyleClasses(List<String> classnames) {
			getWidget().getContent().get(0).getStyleClass().removeAll(classnames);
		}

		@Override
		public void removeStyleClasses(String... classnames) {
			getWidget().getContent().get(0).getStyleClass().removeAll(classnames);
		}

		@Override
		public void setStyleId(String id) {
			getWidget().getContent().get(0).setId(id);
		}

		@Override
		protected Popup createWidget() {
			Popup widget = new Popup();
			widget.autoHideProperty().set(true);
			widget.getContent().add(createPopupContent());
			this.button.setOnAction(this::showPopup);
			this.popup = widget;
			return widget;
		}
		
		private Node createPopupContent() {
			BorderPane borderPane = new BorderPane();
			this.popupLabel = new Label(this.popupLabelText);
			borderPane.setTop(this.popupLabel);
			HBox hbox = new HBox();
			borderPane.setCenter(hbox);
			BorderPane scrollerPane = new BorderPane();
			this.rowsPane = new FlowPane(Orientation.VERTICAL);
			hbox.getChildren().addAll(this.rowsPane, scrollerPane);
			this.currentRow = new FlowPane();
			this.currentRow.setHgap(4.0);
			this.rowsPane.getChildren().add(this.currentRow);
			return borderPane;
		}
		
		protected void showPopup(ActionEvent event) {
			if (!this.popup.isShowing()) {
				Button btn = PopupWMenuImpl.this.button;
				Point2D point = btn.localToScene(0.0, 0.0);
				double x = point.getX() + btn.getScene().getX() + btn.getScene().getWindow().getX() + btn.getWidth();
				double y = point.getY() + btn.getScene().getY() + btn.getScene().getWindow().getY();
				this.popup.show(btn, x, y);
			}
		}

		@Override
		protected void setUserData(WWidgetImpl<Popup, MMenu> widget) {
			getWidget().getContent().get(0).setUserData(widget);
		}

		@Inject
		public void setLabel(@Named(ATTRIBUTE_localizedLabel) String label) {
			this.popupLabelText = label;
		}

		@Override
		public void addElement(WMenuElement<MMenuElement> widget) {
			if (widget instanceof WMenuSeparator) {
				this.currentRow = (FlowPane)widget.getWidget();
				this.rowsPane.getChildren().add(this.currentRow);
			} else {
				this.currentRow.getChildren().add((ButtonBase)widget.getWidget());
			}
		}

		@Override
		public void addElement(int idx, WMenuElement<MMenuElement> widget) {
//			this.currentRow.getChildren().add(idx, (ButtonBase)widget.getWidget());
		}
		

		@Override
		public void removeElement(WMenuElement<MMenuElement> widget) {
//			this.currentRow.getChildren().remove((ButtonBase)widget.getWidget());
		}
		
		@Override
		public void setShowingCallback(Runnable showingCallback) {
			this.showingCallback = showingCallback;
		}
		
		@Override
		public void setHidingCallback(Runnable hidingCallback) {
			this.hidingCallback = hidingCallback;
		}
	}

}