package org.epsilon.tests.workflow;

import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.epsilon.common.parse.problem.ParseProblem;
import org.eclipse.epsilon.emc.emf.EmfModel;
import org.eclipse.epsilon.eol.EolModule;
import org.eclipse.epsilon.eol.IEolExecutableModule;
import org.eclipse.epsilon.eol.exceptions.EolRuntimeException;
import org.eclipse.epsilon.eol.models.IModel;
import org.eclipse.epsilon.etl.EtlModule;

public class EpsilonTask {

	private enum ModuleType {
		  EOL
		, ETL
		, EVL
		, EGL
		, EWL
		, ECL
		, EML
	}

	protected ModuleType moduleType;
	protected String modulePath;
	protected List<IModel> moduleIOModels;
	protected Object result;

	private EpsilonTask() {
		moduleIOModels = new ArrayList<IModel>();
	}
	
	public EpsilonTask(String path) throws EpsilonWorkflowException {
		this();
		modulePath = path;
		moduleType = guessModuleType(path);
	}

	public EpsilonTask(String path, ModuleType type) {
		this();
		modulePath = path;
		moduleType = type;
	}


	private ModuleType guessModuleType(String path) throws EpsilonWorkflowException {
		String extention = path.substring(path.lastIndexOf('.') + 1);
		ModuleType[] types = ModuleType.values();
		for (int i=0; i < types.length; i++) {
			if (types[i].name().equalsIgnoreCase(extention))
				return types[i];
		}
		throw new EpsilonWorkflowException("Could not guess the module type from filename: " + path);
	}

	public IEolExecutableModule createModule() throws EpsilonWorkflowException {
		switch (moduleType) {
		case EOL:
			return new EolModule();
		case ETL:
			return new EtlModule();
		default:
			throw new EpsilonWorkflowException("Cannot handle this kind of modules yet: " + moduleType.name());
		}
	}

	public List<IModel> getModels() {
		return moduleIOModels;
	}
	
	public void addModel(EmfModel model) {
		moduleIOModels.add(model);
	}
	
	
	public String getSourcePath() {
		return modulePath;
	}

	public void execute() throws EpsilonWorkflowException, EolRuntimeException {
		IEolExecutableModule module = createModule();
		try {
			module.parse(EpsilonUtils.getFile(getSourcePath()));
		} catch (URISyntaxException e) {
			throw new EpsilonWorkflowException();
		} catch (Exception e) {
			throw new EpsilonWorkflowException("Failed at parsing module", e);
		}

		if (module.getParseProblems().size() > 0) {
			System.err.println("Parse errors occured...");
			for (ParseProblem problem : module.getParseProblems()) {
				System.err.println(problem.toString());
			}
			System.exit(-1);
		}

		for (IModel model : getModels()) {
			module.getContext().getModelRepository().addModel(model);
		}

		preProcess(module);
		result = module.execute();
		postProcess(module);
		
		module.getContext().getModelRepository().dispose();
	}

	public void preProcess(IEolExecutableModule module) {
		if (module.getOperations().getOperation("rrExecutionModelSingleton") != null)
			System.out.println("ok");
	};
	
	public void postProcess(IEolExecutableModule module) {
	}
}
